#include-once

#CS Library Info/Header
	;        Name/Title:         Opera Browser Automation UDF Library
	;        Description:        Functions to handle some of the common routines for Opera Browser!
	;        Version:            1.01.
	;        Last Update:        5.Jan.2008
	;        Requirements:       AutoIt 3.2.8.1 or higher.
	;        Note:               This UDF Library tested from Opera version 8.54 to Opera version 9.5.
	;        Author:             Copyright  2007 - 2008 CreatoR's Lab (G.Sandler).
	;
	;        ToDo:
	;                        * Add UDFs:
	;                            _OperaSetItem()
	;                            _OperaSetButton()
	;
	;        History Version:
	;                           v1.01 [5.Jan.2008]
	;                           * Now compatible with AutoIt 3.2.10.0.
	;                           * Variable names and spell corrections.
	;                           * Added 2 more UDFs:
	;                                                     _OperaRestart()
	;                                                     _OperaMultiUserModeIsOn()
	;
	;                           v1.0 [28.Dec.2007]
	;                           First Release.
	;
#CE

;===============================================================================
;
; Function Name:  		   _OperaFindDir()
;
; Function Description:    Performs a search for Opera Browser Installed Dir(s).
;
; Parameter(s):            $sInitPath - [Optional] Initial Path to start search from,
;                                          if used -1 (default) then the search performed in all fixed drives (C:\, D:\, etc.).
;                          $iRetCount - [Optional] Defins how many Opera Dirs to return:
;                                          -1 - (default) will return the first found dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return array with full paths to Opera Dir, where:
;                                                $aOperaPaths[0] is the total found Opera dirs.
;                                                $aOperaPaths[N] is the full path to Opera Installed dir.
;                          On Failure -  If no Opera dirs found, Set @error to 1 and return "" (empty string).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaFindDir($sInitPath = -1, $iRetCount = -1)
	Local $aOperaPaths[1]
	Local $aInitPaths[2] = [1, $sInitPath]
	Local $aPaths[1], $hSearch, $sFindNext, $i
	
	If Not __PathIsFolder($sInitPath) Then $aInitPaths = DriveGetDrive("FIXED")
	If @error Then Return SetError(1, 0, "")
	
	For $iP = 1 To UBound($aInitPaths) - 1
		Dim $aPaths[2] = [1, $aInitPaths[$iP]]
		$i = 0
		
		While $i < $aPaths[0]
			$i += 1
			$hSearch = FileFindFirstFile($aPaths[$i] & "\*")
			If $hSearch = -1 Then ContinueLoop
			
			While 1
				$sFindNext = FileFindNextFile($hSearch)
				If @error Then ExitLoop
				
				If $sFindNext = "Opera.exe" Then
					$aOperaPaths[0] += 1
					ReDim $aOperaPaths[$aOperaPaths[0] + 1]
					$aOperaPaths[$aOperaPaths[0]] = $aPaths[$i]
					
					If $iRetCount = -1 Or ($iRetCount > 0 And $aOperaPaths[0] >= $iRetCount) Then ExitLoop 3
				EndIf
				
				$sFindNext = $aPaths[$i] & "\" & $sFindNext
				
				If __PathIsFolder($sFindNext) Then
					$aPaths[0] += 1
					ReDim $aPaths[$aPaths[0] + 1]
					$aPaths[$aPaths[0]] = $sFindNext
				EndIf
			WEnd
			FileClose($hSearch)
		WEnd
	Next
	
	FileClose($hSearch)
	
	If $aOperaPaths[0] < 1 Then Return SetError(1, 0, "")
	Return $aOperaPaths
EndFunc   ;==>_OperaFindDir

;===============================================================================
;
; Function Name:  		   _OperaGetDir()
;
; Function Description:    Returns full path to Opera Browser Installed Dir.
;
; Parameter(s):            None.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Dir, i.e. C:\Program Files\Opera 9.25
;                          On Failure -  Set @error to 1 and return "" (empty string).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetDir()
	Local $sOperaDir = @ScriptDir
	If __PathIsOperaDir($sOperaDir) Then Return $sOperaDir
	
	Local $sOperaDirRegRead = RegRead("HKEY_CURRENT_USER\Software\Opera Software", "Last CommandLine v2")
	Local $aOperaDirArr = StringSplit($sOperaDirRegRead, "\Opera.exe", 1)
	If UBound($aOperaDirArr) > 1 Then $sOperaDir = $aOperaDirArr[1]
	
	If Not __PathIsOperaDir($sOperaDir) Then $sOperaDir = StringTrimRight($sOperaDirRegRead, 10)
	If Not __PathIsOperaDir($sOperaDir) Then $sOperaDir = @ProgramFilesDir & "\Opera"
	
	If Not __PathIsOperaDir($sOperaDir) Then Return SetError(1, 0, "")
	
	Return $sOperaDir
EndFunc   ;==>_OperaGetDir

;===============================================================================
;
; Function Name:  		   _OperaGetVersion()
;
; Function Description:    Returns Opera.exe file Version.
;
; Parameter(s):            $sOpDir - [Optional] Full path to Opera dir,
;                                    so the returned Opera.exe version will be taken from specific Opera installation.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  String with Opera Version, i.e. 9.25.8827.0
;                          On Failure -  Returns 0 and set @error to:
;                                                                      1 - $sOpDir (Opera Dir) not found.
;                                                                      2 - Unable to get Opera version (FileGetVersion() fails).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetVersion($sOpDir = -1)
	If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
	If @error Then Return SetError(1, 0, 0)
	
	Local $sOperaVersion = FileGetVersion($sOpDir & "\Opera.exe")
	If @error Then Return SetError(2, 0, 0)
	
	Return $sOperaVersion
EndFunc   ;==>_OperaGetVersion

;===============================================================================
;
; Function Name:  		   _OperaGetTitle()
;
; Function Description:    Returns Title(s)/Win handle(s) for Opera Window.
;
; Parameter(s):            $sOpDir - [Optional] Full path to Opera dir,
;                                     so the returned Title will be related to specific Opera installation.
;                          $iRet   - [Optional] Defins return type:
;                                     -1 (default) - Will return only Opera Title.
;                                     0 - Will return only Opera Window Handle.
;                                     1 - Will return 2-elements array where [0] is Opera Title, and [1] is Opera Win Handle.
;                                     2 - Will return 2-dimentional array with all founded Opera Windows, where...
;                                              $aOpWinList[0][0] is number of total windows found.
;                                              $aOpWinList[N][0] is Opera Window Title.
;                                              $aOpWinList[N][1] is Opera Window Handle.
;                                           $sOpDir is ignored with this mode.
;                          $iUseDDE - [Optional] Get Opera Title method, if > 0 (0 is the default),
;                                               then the Opera Title will be returned using DDE
;                                                  (for last executed Opera's Window).
;                                               When this method used, $iRet can be only as -1 (default) or as 1
;                                                  (other values are ignored).
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return Opera Title(s)/Window Handle(s) according to passed parameters.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                              1 - $iUseDDE > 0 and there was an error to get access to Opera DDE.
;                                              2 - $iRet = 2 and there was an error to get Opera Windows list.
;                                              3 - Unable to find Opera Title/Win Handle.
;                                              4 - Other error, such as incorrect passed parameter(s) value.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetTitle($sOpDir = -1, $iRet = -1, $iUseDDE = 0)
	Local $sOpTitle = ""
	
	If $iUseDDE Then
		$sOpTitle = _OpenGetURL("", -1, 1)
		If Not IsArray($sOpTitle) Then Return SetError(1, 0, "")
		
		$sOpTitle = StringRegExpReplace($sOpTitle[0], '\A".*?\[(.*)\]"$', '\1')
		
		If $iRet = 1 Then
			Local $aRetArr[2] = [$sOpTitle, WinGetHandle($sOpTitle) ]
			Return $aRetArr
		EndIf
		
		Return $sOpTitle
	EndIf
	
	If $iRet = 2 Then
		Local $aOpWinList = WinList("[REGEXPTITLE:(.*?)- Opera; CLASS:OpWindow]")
		If Not IsArray($aOpWinList) Then Return SetError(2, 0, "")
		Return $aOpWinList
	EndIf
	
	If __PathIsOperaDir($sOpDir) Then
		$sOpTitle = __WinGetTitleEx($sOpDir & "\Opera.exe")
		If Not WinExists($sOpTitle) Or $sOpTitle = "" Then Return SetError(3, 0, "")
		;Return $sOpTitle
	Else
		$sOpDir = _OperaGetDir()
	EndIf
	
	If __PathIsOperaDir($sOpDir) Then $sOpTitle = __WinGetTitleEx($sOpDir & "\Opera.exe")
	
	If Not WinExists($sOpTitle) Or $sOpTitle = "" Then
		$sOpTitle = WinGetTitle("[REGEXPTITLE:(.*?)- Opera; CLASS:OpWindow]")
		$sOpTitle = StringLeft($sOpTitle, StringInStr($sOpTitle, "-", 0, -1))
		If StringRight($sOpTitle, 1) = "-" Then $sOpTitle = StringStripWS(StringTrimRight($sOpTitle, 1), 3)
		If Not WinExists($sOpTitle) Or $sOpTitle = "" Then $sOpTitle = "Opera"
	EndIf
	
	Local $hOphWnd = WinGetHandle($sOpTitle)
	
	Switch $iRet
		Case - 1
			Return $sOpTitle
		Case 0
			Return $hOphWnd
		Case 1
			Local $aRetArr[2] = [$sOpTitle, $hOphWnd]
			Return $aRetArr
	EndSwitch
	Return SetError(4, 0, "")
EndFunc   ;==>_OperaGetTitle

;===============================================================================
;
; Function Name:  		   _OperaGetProfileDir()
;
; Function Description:    Returns full path to Opera Profile Dir.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned profile dir will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Profile Dir,
;                                        i.e. C:\Documents and Settings\User\Application Data\Opera\Opera 9.25\profile
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                          1 - $sOpDir (Opera Dir) not found.
;                                                                          2 - Opera6.ini (main Opera's config file) not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetProfileDir($sOpDir = -1)
	If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
	If @error Then Return SetError(1, 0, "")
	
	Local $sReadMultiUser = IniRead($sOpDir & "\Operadef6.ini", "System", "Multi User", 0)
	If Number($sReadMultiUser) = 1 Then $sOpDir = @AppDataDir & "\Opera\" & StringRegExpReplace($sOpDir, "^.*\\", "")
	
	If Not __PathIsOperaConfigFile($sOpDir & "\profile\Opera6.ini") Then Return SetError(2, 0, "")
	
	Return $sOpDir & "\profile"
EndFunc   ;==>_OperaGetProfileDir

;===============================================================================
;
; Function Name:  		   _OperaGetMailDir()
;
; Function Description:    Returns full path to Opera Mail Dir.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Mail dir will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Mail Dir,
;                                         i.e. C:\Documents and Settings\User\Application Data\Opera\Opera 9.25\mail
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                     1 - $sOpDir (Opera Dir) not found.
;                                                                     2 - Index.ini (main Opera's Mail config file) not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetMailDir($sOpDir = -1)
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpMailDir = IniRead($sOpProfileDir & "\Opera6.ini", "Mail", "Mail Root Directory", "")
	
	If Not __PathIsOperaConfigFile($sOpMailDir & "\Index.ini") Then _
			$sOpMailDir = StringRegExpReplace($sOpProfileDir, "\\[^\\]*$", "") & "\Mail"
	
	If Not __PathIsOperaConfigFile($sOpMailDir & "\Index.ini") Then $sOpMailDir = @ScriptDir & "\Mail"
	If Not __PathIsOperaConfigFile($sOpMailDir & "\Index.ini") Then Return SetError(2, 0, "")
	
	Return $sOpMailDir
EndFunc   ;==>_OperaGetMailDir

;===============================================================================
;
; Function Name:  		   _OperaGetUserJSDir()
;
; Function Description:    Returns full path to Opera main UserJS Dir.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned UserJS dir will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera UserJS Dir,
;                                         i.e. C:\Documents and Settings\User\Application Data\Opera\Opera 9.25\profile\UserJS
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                 1 - $sOpDir (Opera Dir) not found.
;                                                 2 - UserJS dir not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetUserJSDir($sOpDir = -1)
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpUserJSDir = IniRead($sOpProfileDir & "\Opera6.ini", "User Prefs", "User JavaScript File", "")
	
	If Not __PathIsFolder($sOpUserJSDir) Then Return SetError(2, 0, "")
	
	Return $sOpUserJSDir
EndFunc   ;==>_OperaGetUserJSDir

;===============================================================================
;
; Function Name:  		   _OperaGetPluginsDir()
;
; Function Description:    Returns used paths to Opera main Plugins Dir(s).
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Plugins dir will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return array with full paths to Opera Plugins Dir.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                                     1 - $sOpDir (Opera Dir) not found.
;                                                                                     2 - Plugins dir(s) not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetPluginsDir($sOpDir = -1)
	If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpPluginsDir = IniRead($sOpDir & "\Operadef6.ini", "User Prefs", "Plugin Path", "")
	Local $aOpPlugsDir, $aRetOpPlugsDir[1]
	$sOpPluginsDir = StringRegExpReplace($sOpPluginsDir, ";+$", "")
	
	If StringInStr($sOpPluginsDir, ";") Then
		$aOpPlugsDir = StringSplit($sOpPluginsDir, ";")
		
		For $i = 1 To UBound($aOpPlugsDir) - 1
			If __PathIsFolder($aOpPlugsDir[$i]) Then
				$aRetOpPlugsDir[0] += 1
				ReDim $aRetOpPlugsDir[$aRetOpPlugsDir[0] + 1]
				$aRetOpPlugsDir[$aRetOpPlugsDir[0]] = $aOpPlugsDir[$i]
			EndIf
		Next
		
		If $aRetOpPlugsDir[0] > 0 Then Return $aRetOpPlugsDir
	EndIf
	
	If Not __PathIsFolder($sOpPluginsDir) Then $sOpPluginsDir = $sOpDir & "\program\plugins"
	If Not __PathIsFolder($sOpPluginsDir) Then Return SetError(2, 0, "")
	
	Dim $aRetOpPlugsDir[2] = [1, $sOpPluginsDir]
	Return $aRetOpPlugsDir
EndFunc   ;==>_OperaGetPluginsDir

;===============================================================================
;
; Function Name:  		   _OperaGetSkinFile()
;
; Function Description:    Returns used Opera Skin file.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Skin file will be related to specific Opera installation.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Skin file.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                                     1 - $sOpDir (Opera Dir) not found.
;                                                                                     2 - Skin file not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetSkinFile($sOpDir = -1)
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpSkinFile = IniRead($sOpProfileDir & "\Opera6.ini", "User Prefs", "Button Set", "")
	
	If Not __PathIsOperaSkinFile($sOpSkinFile) Then
		If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
		If Not @error Then $sOpSkinFile = $sOpDir & "\skin\standard_skin.zip"
	EndIf
	
	If Not __PathIsOperaSkinFile($sOpSkinFile) Then Return SetError(2, 0, "")
	
	Return $sOpSkinFile
EndFunc   ;==>_OperaGetSkinFile

;===============================================================================
;
; Function Name:  		   _OperaGetLangFile()
;
; Function Description:    Returns used Opera Language file.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Language file will be related to specific Opera installation.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Language file.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                                    1 - $sOpDir (Opera Dir) not found.
;                                                                                    2 - Language file not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetLangFile($sOpDir = -1)
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpLangFile = IniRead($sOpProfileDir & "\Opera6.ini", "User Prefs", "Language File", "")
	
	If Not __PathIsOperaLangFile($sOpLangFile) Then
		If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
		If Not @error Then $sOpLangFile = $sOpDir & "\English.lng"
	EndIf
	
	If Not __PathIsOperaLangFile($sOpLangFile) Then Return SetError(2, 0, "")
	
	Return $sOpLangFile
EndFunc   ;==>_OperaGetLangFile

;===============================================================================
;
; Function Name:  		   _OperaGetConfigFile()
;
; Function Description:    Returns used Opera Configuration file based on given prefix (see Parameter(s)).
;
; Parameter(s):            $sConfig_Prefix - Prefix string that indicate what configurtion file to return, these are supported:
;                                             Menu, Toolbar, Keyboard, Mouse
;                          $sOpDir         - [Optional] Full path to Opera dir,
;                                             so the returned Config file will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success      - Return full path to specific Opera Configuration file.
;                          On Failure      - Returns "" (empty string) and set @error to:
;                                                    1 - $sConfig_Prefix is unsupported prefix string (in this case returns -1).
;                                                    2 - If Opera profile dir not found.
;                                                    3 - Configuration file not found,
;                                                        or founded configuration file is not an Opera Configuration file.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaGetConfigFile($sConfig_Prefix, $sOpDir = -1)
	If Not __PrefixStrIsOperaConfig($sConfig_Prefix) Then Return SetError(1, 0, -1)
	
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(2, 0, "")
	
	Local $sOpConfigFile = IniRead($sOpProfileDir & "\Opera6.ini", "User Prefs", $sConfig_Prefix & " Configuration", "")
	
	If Not __PathIsOperaConfigFile($sOpConfigFile) Then
		If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
		If Not @error Then $sOpConfigFile = $sOpDir & "\defaults\standard_" & $sConfig_Prefix & ".ini"
	EndIf
	
	If Not __PathIsOperaConfigFile($sOpConfigFile) Then Return SetError(3, 0, "")
	
	Return $sOpConfigFile
EndFunc   ;==>_OperaGetConfigFile

;===============================================================================
;
; Function Name:  		   _OperaSetConfigFile()
;
; Function Description:    Set specific Opera Configuration file based on given prefix (see Parameter(s)).
;
; Parameter(s):            $sConfig_Prefix - Prefix string that indicate what configurtion file to set, these are supported:
;                                             Menu, Toolbar, Keyboard, Mouse
;                          $sConfig_Name   - Name for the configuration file, used as file name and as menu info name.
;                          $sSetConfigFile - [Optional] Full path to external configuration file,
;                                             this file will be used as main configuration file,
;                                             if not found, standard file will be cloned to config dir with name = $sConfig_Name.
;                          $sOpDir         - [Optional] Full path to Opera dir,
;                                             so the returned Config file will be related to specific Opera installed dir.
;                          $iRepRenFlag    - [Optional] Defines replace mode:
;                                             -1 (default) or 0 will backup existing file in case the standard file is copied.
;                                                (*Without* replacing the existing backup file).
;                                             1 will backup existing file in case the standard file is copied.
;                                                (*Will* replace the existing backup file).
;                                             2 auto-rename existing file in case the standard file is copied.
;                                             In other cases existing file will be replaced.
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success - Set specific Opera Configuration file for usage in Opera Browser,
;                                         and return full path to "old" (used) Opera specific Configuration file.
;                          On Failure - Returns "" (empty string) and set @error to:
;                                                    1 - $sConfig_Prefix is unsupported prefix string (in this case returns -1).
;                                                    2 - Opera profile dir not found.
;                                                    3 - Current Configuration file = $sSetConfigFile.
;                                                    4 - $sSetConfigFile is not found (or it is not an Opera config file),
;                                                         and Opera Dir not found.
;                                                    5 - Unable to copy Configuration file (if standard config file cloned).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaSetConfigFile($sConfig_Prefix, $sConfig_Name, $sSetConfigFile = -1, $sOpDir = -1, $iRepRenFlag = -1)
	If Not __PrefixStrIsOperaConfig($sConfig_Prefix) Then Return SetError(1, 0, -1)
	
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(2, 0, "")
	
	Local $sCurrConfigFile = _OperaGetConfigFile($sConfig_Prefix, $sOpDir)
	If Not @error And $sSetConfigFile = $sCurrConfigFile Then Return SetError(3, 0, $sCurrConfigFile)
	
	If StringRight($sConfig_Name, 4) <> ".ini" Then $sConfig_Name &= ".ini"
	
	If Not __PathIsOperaConfigFile($sSetConfigFile) Then
		If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
		If @error Then Return SetError(4, 0, "")
		
		$sStandard_ConfigFile = $sOpDir & "\defaults\standard_" & $sConfig_Prefix & ".ini"
		
		If $iRepRenFlag = 2 Then $sConfig_Name = __FileNewName($sConfig_Name, $sOpProfileDir & "\" & $sConfig_Prefix)
		$sSetConfigFile = $sOpProfileDir & "\" & $sConfig_Prefix & "\" & $sConfig_Name
		
		Switch $iRepRenFlag
			Case Default, -1, 0
				__Backup_Config($sSetConfigFile, 0)
			Case 1
				__Backup_Config($sSetConfigFile, 1)
		EndSwitch
		
		If Not FileCopy($sStandard_ConfigFile, $sSetConfigFile, 9) Then Return SetError(5, 0, "")
	EndIf
	
	IniWrite($sOpProfileDir & "\Opera6.ini", "User Prefs", $sConfig_Prefix & " Configuration", $sSetConfigFile)
	IniWrite($sSetConfigFile, "Info", "NAME", StringRegExpReplace($sConfig_Name, "(?i).ini+$", ""))
	
	Return $sCurrConfigFile
EndFunc   ;==>_OperaSetConfigFile

;===============================================================================
;
; Function Name:  		   _OperaSetSkinFile()
;
; Function Description:    Set Opera Skin file.
;
; Parameter(s):            $sOpSkinFile  - Full path to skin file that needs to be set.
;                          $sOpDir       - [Optional] Full path to Opera dir,
;                                           so the Skin file will be set for specific Opera installation.
;                          $iCopyToSkins - [Optional] If this is <> 0 then skin file will be copied to Opera's Skin folder.
;                          $iRepRenFlag  - [Optional] Defines replace mode:
;                                           -1 (default) or 0 will backup existing file in case the skin file is copied.
;                                              (*Without* replacing the existing backup file).
;                                           1 will backup existing file in case the skin file is copied.
;                                              (*Will* replace the existing backup file).
;                                           2 auto-rename existing file in case the skin file is copied.
;                                           In other cases existing file will be replaced.
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success -  Set Opera Skin file, and return full path to "old" (used) Opera Skin file.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                   1 - Opera profile dir not found.
;                                                                   2 - Current Skin file = $sOpSkinFile.
;                                                                   3 - $sOpSkinFile not found, or it's not an Opera Skin file.
;                                                                   4 - Unable to copy skin file (with $iCopyToSkins mode on).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaSetSkinFile($sOpSkinFile, $sOpDir = -1, $iCopyToSkins = 0, $iRepRenFlag = -1)
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sCurrOpSkinFile = _OperaGetSkinFile($sOpDir)
	If Not @error And $sCurrOpSkinFile = $sOpSkinFile Then Return SetError(2, 0, "")
	
	If Not __PathIsOperaSkinFile($sOpSkinFile) Then Return SetError(3, 0, "")
	
	If $iCopyToSkins <> 0 Then
		Local $sOpSkinName = StringRegExpReplace($sOpSkinFile, "^.*\\", "")
		Local $sOpSkinPath = $sOpProfileDir & "\Skin"
		Local $sOpDestSkinFile = $sOpSkinPath & "\" & $sOpSkinName
		
		Switch $iRepRenFlag
			Case Default, -1, 0
				__Backup_Config($sOpDestSkinFile, 0)
			Case 1
				__Backup_Config($sOpDestSkinFile, 9)
			Case 2
				$sOpDestSkinFile = $sOpSkinPath & "\" & __FileNewName($sOpSkinName, $sOpSkinPath)
		EndSwitch
		
		If Not FileCopy($sOpSkinFile, $sOpDestSkinFile, 9) Then SetError(4, 0, "")
		$sOpSkinFile = $sOpDestSkinFile
	EndIf
	
	IniWrite($sOpProfileDir & "\Opera6.ini", "User Prefs", "Button Set", $sOpSkinFile)
	
	Return $sCurrOpSkinFile
EndFunc   ;==>_OperaSetSkinFile

;===============================================================================
;
; Function Name:  		   _OperaSetUserJSDir()
;
; Function Description:    Set Opera UserJS directory.
;
; Parameter(s):            $sOpUserJSDir  - Full path to User JavaScript directory (that include UserJS scripts).
;                          $sOpDir        - [Optional] Full path to Opera dir,
;                                            so the UserJS directory will be set for specific Opera installation.
;                          $iEnableUserJS - [Optional] If this is = 1 (default) then UserJS will be enabled for Opera Browser.
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success -  Set Opera UserJS directory, and return full path to "old" (used) Opera UserJS directory.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                             1 - Opera profile dir not found.
;                                                                             2 - Current UserJS dir = $sOpUserJSDir.
;                                                                             3 - $sOpUserJSDir is not an existing directory.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaSetUserJSDir($sOpUserJSDir, $sOpDir = -1, $iEnableUserJS = 1)
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sCurrOpUserJSDir = _OperaGetUserJSDir($sOpDir)
	If Not @error And $sCurrOpUserJSDir = $sOpUserJSDir Then Return SetError(2, 0, "")
	
	If Not __PathIsFolder($sOpUserJSDir) Then Return SetError(3, 0, "")
	
	IniWrite($sOpProfileDir & "\Opera6.ini", "User Prefs", "User JavaScript File", $sOpUserJSDir)
	
	If $iEnableUserJS = 1 Then
		IniWrite($sOpProfileDir & "\Opera6.ini", "User Prefs", "Browser JavaScript", 2)
		IniWrite($sOpProfileDir & "\Opera6.ini", "User Prefs", "User JavaScript", 1)
		IniWrite($sOpProfileDir & "\Opera6.ini", "User Prefs", "Always Load User JavaScript", 1)
	EndIf
	
	Return $sCurrOpUserJSDir
EndFunc   ;==>_OperaSetUserJSDir

;===============================================================================
;
; Function Name:  		   _OperaSetLangFile()
;
; Function Description:    Set Opera Language file.
;
; Parameter(s):            $sOpLangFile  - Full path to Language file that needs to be set.
;                          $sOpDir       - [Optional] Full path to Opera dir,
;                                           so the Language file will be set for specific Opera installation.
;                          $iCopyToOpDir - [Optional] If this is <> 0 then language file will be copied to Opera's folder.
;                          $iRepRenFlag  - [Optional] Defines replace mode:
;                                           -1 - (default) or 0 will backup existing file in case the lang file is copied.
;                                                 (*Without* replacing the existing backup file).
;                                           1  - will backup existing file in case the lang file is copied.
;                                              (*Will* replace the existing backup file).
;                                           2  - auto-rename existing file in case the lang file is copied.
;                                           * In other cases existing file will be replaced.
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success -  Set Opera Language file, and return full path to "old" (used) Opera Language file.
;                          On Failure -  Returns "" (empty string),
;                                           and set @error to:
;                                                              1 - Opera profile dir not found.
;                                                              2 - Current Language file = $sOpLangFile.
;                                                              3 - $sOpLangFile not found, or it's not an Opera Language file.
;                                                              4 - $sOpDir (Opera dir) not found.
;                                                              5 - Unable to copy language file (with $iCopyToOpDir mode on).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaSetLangFile($sOpLangFile, $sOpDir = -1, $iCopyToOpDir = 0, $iRepRenFlag = -1)
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sCurrOpLangFile = _OperaGetLangFile($sOpDir)
	If Not @error And $sCurrOpLangFile = $sOpLangFile Then Return SetError(2, 0, "")
	
	If Not __PathIsOperaLangFile($sOpLangFile) Then Return SetError(3, 0, "")
	
	If $iCopyToOpDir <> 0 Then
		If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
		If @error Then Return SetError(4, 0, "")
		
		Local $sOpLangName = StringRegExpReplace($sOpLangFile, "^.*\\", "")
		Local $sOpDestLangFile = $sOpDir & "\" & $sOpLangName
		
		Switch $iRepRenFlag
			Case Default, -1, 0
				__Backup_Config($sOpDestLangFile, 0)
			Case 1
				__Backup_Config($sOpDestLangFile, 9)
			Case 2
				$sOpDestLangFile = $sOpDir & "\" & __FileNewName($sOpLangName, $sOpDir)
		EndSwitch
		
		If Not FileCopy($sOpLangFile, $sOpDestLangFile, 9) Then SetError(5, 0, "")
		$sOpLangFile = $sOpDestLangFile
	EndIf
	
	IniWrite($sOpProfileDir & "\Opera6.ini", "User Prefs", "Language File", $sOpLangFile)
	
	Return $sCurrOpLangFile
EndFunc   ;==>_OperaSetLangFile

;===============================================================================
;
; Function Name:  		   _OperaSelectFolder()
;
; Function Description:    Open Browse Dialog to select Opera install dir.
;
; Parameter(s):            $sTitle      - Title for FileSelectFolder dialog.
;                          $sErrMsgText - Message string to show when selected wrong path to Opera Browser.
;                          $sOpDir      - [Optional] Full path to initial Opera dir, to start FileSelectFolder dialog with.
;                          $iInputID    - [Optional] Identifier to GUI Edit control, so the selected path will be updated in it.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success   -  Return new selected Opera Dir.
;                          On Failure   -  Set @error to 1 and return old initial Opera dir ($sOpDir).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaSelectFolder($sTitle, $sErrMsgText, $sOpDir = "", $iInputID = 0)
	Local $sSelectOpDir, $sInitDir = $sOpDir
	
	While 1
		$sSelectOpDir = FileSelectFolder($sTitle, "", 2, $sInitDir)
		If @error Then Return SetError(1, 0, $sOpDir)
		
		If Not __PathIsOperaDir($sSelectOpDir) Then
			MsgBox(262144 + 48, "Attention", $sErrMsgText)
			$sInitDir = $sSelectOpDir
			ContinueLoop
		EndIf
		
		If $iInputID > 0 Then GUICtrlSetData($iInputID, $sSelectOpDir)
		Return $sSelectOpDir
	WEnd
EndFunc   ;==>_OperaSelectFolder

;===============================================================================
;
; Function Name:  		   _OperaIsRuning()
;
; Function Description:    Check if Opera Browser process is currently runing.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        to check the run state for the specific Opera executable file.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return True.
;                          On Failure -  Returns False and set @error to 1 (even if unable check Opera's "Run State").
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaIsRuning($sOpDir = -1)
	If Not ProcessExists("Opera.exe") Then Return False
	
	If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
	If __PathIsOperaDir($sOpDir) And __ProcessPathExists($sOpDir & "\Opera.exe") Then Return True
	
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $nRunState = IniRead($sOpProfileDir & "\Opera6.ini", "State", "Run", "Error")
	If $nRunState = "Error" Then Return SetError(1, 0, False)
	
	Return Not Number($nRunState) = 0
EndFunc   ;==>_OperaIsRuning

;===============================================================================
;
; Function Name:  		   _OperaRun()
;
; Function Description:    Run Opera Browser.
;
; Parameter(s):            $sOpExec      - [Optional] Full path to Opera executable file.
;                          $vArguments  -  [Optional] Arguments to pass when executing Opera.exe, can be a *one*-dimentional array.
;                          $sWorkDir     - [Optional] Set Working directory when executing Opera.exe
;                                            (-1 is the default, will set "Opera Dir" as working dir).
;                          $iState       - [Optional] The "show" flag of the executed program, default is @SW_SHOWNORMAL.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success    - Return 1 if Opera.exe was executed.
;                          On Failure    - Returns 0 (zero) and set @error to:
;                                                                              1 - $sOpDir (Opera Dir) not found.
;                                                                              2 - Unable to execute Opera.exe.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaRun($sOpExec = -1, $vArguments = "", $sWorkDir = -1, $iState = @SW_SHOWNORMAL)
	If Not FileExists($sOpExec) Or StringRight($sOpExec, 4) <> ".exe" Then $sOpExec = _OperaGetDir() & "\Opera.exe"
	If @error Then Return SetError(1, 0, "")
	
	Local $iOld_Opt_REF = Opt("RunErrorsFatal", 0)
	
	If IsArray($vArguments) And UBound($vArguments, 0) = 1 Then
		Local $aSwitchArr = $vArguments
		Local $sSpace = " ", $iUbound = UBound($aSwitchArr) - 1
		$vArguments = ""
		For $i = 1 To $iUbound
			If $i = $iUbound Then $sSpace = ""
			$vArguments &= $aSwitchArr[$i] & $sSpace
		Next
	EndIf
	
	If $sWorkDir = -1 Or $sWorkDir = Default Then $sWorkDir = StringRegExpReplace($sOpExec, "\\[^\\]*$", "")
	
	ShellExecute($sOpExec, $vArguments, $sWorkDir, "open", $iState)
	Local $iError = @error
	
	Opt("RunErrorsFatal", $iOld_Opt_REF)
	
	If $iError Then Return SetError(2, 0, 0)
	Return 1
EndFunc   ;==>_OperaRun

;===============================================================================
;
; Function Name:  		   _OperaClose()
;
; Function Description:    Close Opera Browser.
;
; Parameter(s):            $sOpDir       - [Optional] Full path to Opera dir,
;                                           so the specific Opera window(s) will be closed.
;                          $iInstance    - [Optional] Defins how many Opera instances to close:
;                                           0 (default) - will close instanse of Opera related to given/found Opera Dir.
;                                           -1  - will close all found Opera instanses.
;                                           > 0 - will close this much Opera instanses.
;                          $iForceClose  - [Optional] If <> 0 then it will force to close Opera window(s),
;                                           even if there will be a confirm close message (0 is the default).
;
; Requirement(s):          None.
;
; Return Value(s):         On Success    - Return 1 regardless if Opera was closed or not.
;                          On Failure    - Return 0 and set @error to:
;                                                                      1 - $sOpDir (Opera Dir) not found.
;                                                                      2 - Unable to get Opera title(s).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaClose($sOpDir = -1, $iInstance = 0, $iForceClose = 0)
	If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
	If @error Then Return SetError(1, 0, 0)
	
	Switch $iInstance
		Case 0
			Local $hOphWnd = _OperaGetTitle($sOpDir, 0)
			If @error Then Return SetError(2, 0, 0)
			
			Local $sConfirmExitWinExists
			
			While WinExists($hOphWnd) Or __ProcessPathExists($sOpDir & "\Opera.exe")
				If Not WinExists($hOphWnd) Then
					$hOphWnd = _OperaGetTitle($sOpDir, 0)
					If @error Then Return 1
				EndIf
				
				WinActivate($hOphWnd)
				WinClose($hOphWnd)
				
				If $iForceClose <> 0 Then WinWaitClose($hOphWnd, "", 1)
				
				If $iForceClose <> 0 And WinExists($hOphWnd) Then
					$sConfirmExitWinExists = __ConfirmExitOperaWinExists($sOpDir)
					
					If $sConfirmExitWinExists <> False Then
						WinActivate($sConfirmExitWinExists)
						WinWaitActive($sConfirmExitWinExists, "", 1)
						ControlSend($sConfirmExitWinExists, "", "", "{ENTER}")
						Sleep(200)
					Else
						WinClose($hOphWnd)
					EndIf
				EndIf
			WEnd
		Case Else
			Local $sConfirmExitWinExists, $iWinProc, $sProcPath
			Local $aOpTitles = _OperaGetTitle("", 2)
			If @error Then Return SetError(2, 0, 0)
			
			If $iInstance < 0 Then $iInstance = $aOpTitles[0][0]
			
			For $i = 1 To $iInstance
				If $i > $aOpTitles[0][0] Then ExitLoop
				WinClose($aOpTitles[$i][1])
				If $iForceClose <> 0 Then WinWaitClose($aOpTitles[$i][1], "", 1)
				
				If $iForceClose <> 0 And WinExists($aOpTitles[$i][1]) Then
					$iWinProc = WinGetProcess($aOpTitles[$i][1])
					$sProcPath = StringRegExpReplace(__ProcessGetPath($iWinProc), "\\[^\\]*$", "")
					
					While WinExists($aOpTitles[$i][1]) Or ProcessExists($iWinProc)
						$sConfirmExitWinExists = __ConfirmExitOperaWinExists($sProcPath)
						
						If $sConfirmExitWinExists <> False Then
							WinActivate($sConfirmExitWinExists)
							ControlSend($sConfirmExitWinExists, "", "", "{ENTER}")
							Sleep(200)
						Else
							WinClose($aOpTitles[$i][1])
						EndIf
					WEnd
				EndIf
			Next
	EndSwitch
	
	Return 1
EndFunc   ;==>_OperaClose

;===============================================================================
;
; Function Name:  		   _OperaKill()
;
; Function Description:    Kill Opera Browser.
;
; Parameter(s):            $sOpDir       - [Optional] Full path to Opera dir,
;                                           so the specific Opera process (that executed from Opera dir) will be killed.
;                          $iInstance    - [Optional] Defins how many Opera instances to kill:
;                                           < 1 - (0 is the default) will kill instanse of Opera related to given/found Opera Dir.
;                                           > 0 - will kill this much Opera instanses (processes).
;                          $iWaitClose   - [Optional] If > 0 then function will wait this much time (seconds)
;                                                            until Opera process is closed.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success    - Return 1 regardless if Opera process was killed or not.
;                          On Failure    - Returns 0
;                                             and set @error to:
;                                                                 1 - If $iInstance >= 1 and unable to get Opera process(es).
;                                                                 2 - If $sOpDir (Opera Dir) not found (when $iInstance < 1).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaKill($sOpDir = -1, $iInstance = 0, $iWaitClose = 0)
	Select
		Case $iInstance >= 1
			Local $aOperaPIDs = __GetOperaPIDs()
			If @error Then Return SetError(1, 0, 0)
			
			For $i = 1 To $aOperaPIDs[0]
				ProcessClose($aOperaPIDs[$i], 1)
				If $iWaitClose > 0 Then ProcessWaitClose($aOperaPIDs[$i], $iWaitClose)
				If $iInstance = $i Then Return 1
			Next
		Case Else
			If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
			If @error Then Return SetError(1, 0, 0)
			
			Local $iOperaPID = __ProcessPathExists($sOpDir & "\Opera.exe")
			If @error Then Return SetError(2, 0, 0)
			
			ProcessClose($iOperaPID, 1)
			If $iWaitClose > 0 Then ProcessWaitClose($iOperaPID, $iWaitClose)
	EndSelect
	
	Return 1
EndFunc   ;==>_OperaKill

;===============================================================================
;
; Function Name:  		   _OperaRestart()
;
; Function Description:    Restarts Opera Browser.
;
; Parameter(s):            $sOpDir       - [Optional] Full path to Opera dir, to restart specific Opera instance.
;                          $iForceClose  - [Optional] If <> 0 then it will force to close Opera window(s),
;                                           even if there will be a confirm close message (0 is the default).
;
; Requirement(s):          None.
;
; Return Value(s):         On Success    - Return 1 if Opera restarted.
;                          On Failure    - Returns 0 (zero) and set @error to:
;                                                                     1 - $sOpDir (Opera Dir) not found.
;                                                                     2 - Unable to close runing Opera.exe process.
;                                                                     3 - Unable to execute Opera, error by _OperaRun() function.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaRestart($sOpDir = -1, $iForceClose = 0)
	If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
	If @error Then Return SetError(1, 0, 0)
	
	If ProcessExists("Opera.exe") Then
		_OperaClose($sOpDir, 0, $iForceClose)
		If @error Then Return SetError(2, 0, 0)
	EndIf
	
	If __ProcessPathExists($sOpDir & "\Opera.exe") Then Return SetError(2, 0, 0)
	
	_OperaRun($sOpDir & "\Opera.exe")
	If @error Then Return SetError(3, 0, 0)
	
	Return 1
EndFunc   ;==>_OperaRestart

;===============================================================================
;
; Function Name:  		   _OperaMultiUserModeIsOn()
;
; Function Description:    Check if Opera installed with Multi User profile.
;
; Parameter(s):            $sOpDir         - [Optional] Full path to Opera dir,
;                                             to check for specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success      - Returns True if Opera installed with Multi User profile, otherwise returns False.
;                          On Failure      - Returns "" (empty string) and set @error to:
;                                                    1 - Opera dir not found.
;                                                    2 - Unable to read Opera's configuration file.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _OperaMultiUserModeIsOn($sOpDir = -1)
	If Not __PathIsOperaDir($sOpDir) Then $sOpDir = _OperaGetDir()
	If @error Then Return SetError(1, 0, "")
	
	Local $sReadMultiUser = IniRead($sOpDir & "\Operadef6.ini", "System", "Multi User", "Error")
	If $sReadMultiUser = "Error" Then Return SetError(2, 0, "")
	
	Return Number($sReadMultiUser) = 1
EndFunc   ;==>_OperaMultiUserModeIsOn

;===============================================================================
;
; Function Name:  		   _Opera_Integration_Example()
;
; Function Description:    Show an Example (a "template" if you like ;) ) on how to integrate your app to Opera Browser.
;
; Parameter(s):            None.
;
; Requirement(s):          None.
;
; Return Value(s):         None.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru)
;
;=====================================================================
Func _Opera_Integration_Example()
	#include <GuiConstants.au3>
	
	Local $OperaDir, $OperaProfileDir, $OpTitle
	Local $AppName, $AskIntegr, $OpPathReadInput
	Local $Intgr_GUI, $OperaPath_Input, $Browse_Button, $OK_Button, $Cancel_Button
	
	;----- Set Opera Preferences -----
	$OperaDir = _OperaGetDir()
	$OperaProfileDir = _OperaGetProfileDir($OperaDir)
	$AppName = "@Opera Integrator Example@"
	$OpTitle = _OperaGetTitle($OperaDir)
	
	;----- Integration GUI -----
	$AskIntegr = MsgBox(262144 + 36, _
			$AppName & " - Integration to Opera", _
			"Would you like to integrate this program into Opera Browser?")
	If $AskIntegr <> 6 Then Return

	$Intgr_GUI = GUICreate($AppName & " - Integration to Opera", 380, 190)
	
	$OperaPath_Input = GUICtrlCreateInput($OperaDir, 20, 120, 320, -1, -1, $SS_BLACKFRAME)
	GUICtrlSetTip(-1, "Path to browser Opera")
	
	$Browse_Button = GUICtrlCreateButton("...", 345, 120, 20, 20)
	
	GUICtrlCreateLabel("Type (or select) the path to Opera Browser:", 77, 50, -1, 50, $SS_CENTER)
	GUICtrlSetFont(-1, 12, 400, 0, "Comic Sans MS")
	
	GUICtrlCreateIcon($OperaDir & "\Opera.exe", 0, 175, 13)
	
	$OK_Button = GUICtrlCreateButton("OK", 110, 150, 50)
	$Cancel_Button = GUICtrlCreateButton("Cancel", 230, 150, 50)
	
	GUISetState(@SW_SHOW, $Intgr_GUI)
	
	While 1
		Switch GUIGetMsg()
			Case $Browse_Button
				$OpPathReadInput = GUICtrlRead($OperaPath_Input)
				GUISetState(@SW_HIDE, $Intgr_GUI)
				$OperaDir = _OperaSelectFolder("Please Select Opera dir", _
						"Can not find Opera files (Opera.exe), please select folder with Opera Browser...", _
						$OpPathReadInput, $OperaPath_Input)
				GUISetState(@SW_SHOW, $Intgr_GUI)
			Case $OK_Button
				$OpPathReadInput = GUICtrlRead($OperaPath_Input)
				If Not __PathIsOperaDir($OpPathReadInput) Then
					MsgBox(262144 + 48, "Attention", _
							"Can not find Opera files (Opera.exe), please select folder with Opera Browser...")
					GUISetState(@SW_HIDE, $Intgr_GUI)
					$OperaDir = _OperaSelectFolder("Please Select Opera dir", _
							"Can not find Opera files (Opera.exe), please select folder with Opera Browser...", _
							$OpPathReadInput, $OperaPath_Input)
					GUISetState(@SW_SHOW, $Intgr_GUI)
				EndIf
				If FileExists($OpPathReadInput & "\Opera.exe") Then
					GUIDelete($Intgr_GUI)
					;Integration_Proc()
					;Here is should be whatever you like to do when user ready to integrate your Application to Opera.
					MsgBox(262144 + 64, $AppName & " - Finishing..", "Integration to Opera is successful completed!")
					ExitLoop
				EndIf
			Case $GUI_EVENT_CLOSE, $Cancel_Button
				GUIDelete($Intgr_GUI)
				ExitLoop
		EndSwitch
	WEnd
EndFunc   ;==>_Opera_Integration_Example

;===============================================================================
;
; Function Name:           OpenGetURL()
; Function Description:    Open/Get URL in/from certain browser using DDE.
;
; Parameter(s):            $sURL     - [Optional] Address of page to open,
;                                       if this is empty string (default), then returned Url from address field of particular tab.
;                          $iWin     - [Optional] Number of tab:
;                                       -1 - Current tab.
;                                       0  - New tab (when opening).
;                          $iRetType - [Optional] Definds returned value:
;                                        0  - String with Title and URL address.
;                                        1  - Array with 3 elements...
;                                             $aRetArr[0] = Title
;                                             $aRetArr[1] = URL address
;                                             $aRetArr[2] = String with Title And URL address.
;                          $sServer  - [Optional] Serever to open/get Url in/from.
;                          $iWait    - [Optional] Waiting Timeout in milliseconds, on overload will return an error.
;
; Requirement(s):          None
; Return Value(s):         On Success -  See 'Parameter(s)'.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                      1 - Error to open Dll (user32.dll)
;                                                      2 - Error Initializing DDE
;                                                       (@extended includes more details about returned value from DllCall).
;                                                      3 - Other DDE Errors
;                                                       (@extended macro include more details about returned value from DllCall).
; Author(s):               amel27 (A.Melnichuk)
;
;=====================================================================
Func _OpenGetURL($sURL = "", $iWin = -1, $iRetType = 0, $sServer = "Opera", $iWait = 10000)
	Local $aRet, $uIdInst = DllStructCreate("int")
	Local $hServer[1], $hTopic[1], $hItem[1], $hConv[1], $hData[1], $sData[1]
	Local $sTopic = "WWW_OpenURL", $sItem = $sURL & ',,0x' & Hex($iWin)
	If $sURL = '' Then
		$sTopic = "WWW_GetWindowInfo"
		$sItem = "0x" & Hex($iWin)
	EndIf
	Local $hDll = DllOpen("user32.dll")
	If $hDll = -1 Then Return SetError(1, 0, "") ; Error to open Dll
	$aRet = DllCall("user32.dll", "int", "DdeInitialize", "ptr", DllStructGetPtr($uIdInst), "ptr", 0, "int", 0, "int", 0)
	If $aRet[0] Then Return SetError(2, $aRet[0], "") ; Error Initializing DDE
	$hServer = DllCall($hDll, "int", "DdeCreateStringHandle", "int", DllStructGetData($uIdInst, 1), "str", $sServer, "int", 1004)
	If $hServer[0] Then
		$hTopic = DllCall($hDll, "int", "DdeCreateStringHandle", "int", DllStructGetData($uIdInst, 1), "str", $sTopic, "int", 1004)
		If $hTopic[0] Then
			$hItem = DllCall($hDll, "int", "DdeCreateStringHandle", _
					"int", DllStructGetData($uIdInst, 1), "str", $sItem, "int", 1004)
			If $hItem[0] Then
				$hConv = DllCall($hDll, "int", "DdeConnect", "int", _
						DllStructGetData($uIdInst, 1), "int", $hServer[0], "int", $hTopic[0], "int", 0)
				If $hConv[0] Then
					$hData = DllCall($hDll, "int", "DdeClientTransaction", "ptr", 0, "int", 0, _
							"int", $hConv[0], "int", $hItem[0], "int", 1, "int", 0x20B0, "int", $iWait, "ptr", 0)
					If $hData[0] Then $sData = DllCall($hDll, "str", "DdeAccessData", "int", $hData[0], "ptr", 0)
				EndIf
			EndIf
		EndIf
	EndIf
	$iErr = DllCall($hDll, "int", "DdeGetLastError", "int", DllStructGetData($uIdInst, 1))
	If $hData[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hData[0])
	If $hConv[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hConv[0])
	If $hItem[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hItem[0])
	If $hTopic[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hTopic[0])
	If $hServer[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hServer[0])
	If $iErr[0] Then Return SetError(3, $iErr[0], "") ; Othe DDE Errors
	DllCall($hDll, "int", "DdeUninitialize", "int", DllStructGetData($uIdInst, 1))
	DllClose($hDll)
	
	If StringRight($sData[0], 3) = ',""' Then $sData[0] = StringTrimRight($sData[0], 3)
	If $sURL = '' Then $sURL = StringRegExpReplace($sData[0], '^"([^"]*?)".*', '"\1"')
	
	If $iRetType = 1 Then
		Local $sRetTitle = StringReplace(StringTrimLeft($sData[0], StringLen($sURL) + 1), '\"', '"')
		Local $aRetArr[3] = [$sRetTitle, $sURL, StringReplace($sData[0], '\"', '"') ]
		Return $aRetArr
	EndIf
	
	Return $sURL
EndFunc   ;==>_OpenGetURL


#Region ================================ HELPER FUNCS ================================
;Extended function to return Title
Func __WinGetTitleEx($sExTitle)
	If IsHWnd($sExTitle) Then Return WinGetTitle($sExTitle)
	Local $iOld_Opt_WTMM = Opt("WinTitleMatchMode", 2)
	Local $sRetTitle = __ProcessPathExists($sExTitle)
	If $sRetTitle Then $sRetTitle = __WinGetTitleByPID($sRetTitle)
	If Not $sRetTitle Then $RetTitle = WinGetTitle($sExTitle)
	If Not WinExists($sRetTitle) Or Not $sRetTitle Then
		Opt("WinTitleMatchMode", 4)
		$sRetTitle = WinGetTitle($sExTitle)
		If Not $sRetTitle Then $sRetTitle = WinGetTitle("[REGEXPTITLE:(?i)(.*)" & $sExTitle & "(.*)]")
	EndIf
	Opt("WinTitleMatchMode", $iOld_Opt_WTMM)
	Return $sRetTitle
EndFunc   ;==>__WinGetTitleEx

;Retrieves Title base on given PID (Process ID)
;Based on Smoke_N's _WinGetHandleByPID()
Func __WinGetTitleByPID($iPID, $iRetMode = 1)
	If IsString($iPID) Then $iPID = ProcessExists($iPID)
	Local $aWList = WinList(), $sHold
	For $iCC = 1 To $aWList[0][0]
		If WinGetProcess($aWList[$iCC][1]) = $iPID And BitAND(WinGetState($aWList[$iCC][1]), 2) Then
			If $iRetMode = 1 Then Return $aWList[$iCC][0]
			$sHold &= $aWList[$iCC][0] & Chr(1)
		EndIf
	Next
	If $sHold Then Return StringSplit(StringTrimRight($sHold, 1), Chr(1))
	Return SetError(1, 0, 0)
EndFunc   ;==>__WinGetTitleByPID

;Check if given path is a runing process and return PID of that process.
Func __ProcessPathExists($sPath)
	If Not FileExists($sPath) Then Return SetError(1, 0, False)
	
	Local $sPathName = StringRegExpReplace($sPath, "^.*\\", "")
	Local $aProcList = ProcessList($sPathName)
	
	If Not ProcessExists($sPathName) Then Return SetError(2, 0, False)
	Local $iUbound = UBound($aProcList) - 1
	
	Local $aProc, $aPath
	Local $vStruct = DllStructCreate('int[1024]')
	
	For $i = 1 To $iUbound
		$aProc = DllCall('kernel32.dll', 'hwnd', 'OpenProcess', 'int', BitOR(0x0400, 0x0010), 'int', 0, 'int', $aProcList[$i][1])
		If Not IsArray($aProc) Or Not $aProc[0] Then Return SetError(3, 0, False)
		
		DllCall('Psapi.dll', 'int', 'EnumProcessModules', _
				'hwnd', $aProc[0], _
				'ptr', DllStructGetPtr($vStruct), _
				'int', DllStructGetSize($vStruct), _
				'int*', 0)
		
		$aPath = DllCall('psapi.dll', 'int', 'GetModuleFileNameEx', _
				'hwnd', $aProc[0], _
				'int', DllStructGetData($vStruct, 1), _
				'str', '', _
				'int', 2048)
		If IsArray($aPath) And $aPath[3] = $sPath Then Return SetExtended($iUbound, $aProcList[$i][1])
	Next
	
	Return SetExtended($iUbound, False)
EndFunc   ;==>__ProcessPathExists

;Retrieves Path based on given PID (Process ID)
Func __ProcessGetPath($iPID)
	If StringRight($iPID, 3) = "exe" Then $iPID = ProcessExists($iPID)
	If Not ProcessExists($iPID) Then Return SetError(1, 0, "")
	
	Local $aProc = DllCall('kernel32.dll', 'hwnd', 'OpenProcess', 'int', BitOR(0x0400, 0x0010), 'int', 0, 'int', $iPID)
	If Not IsArray($aProc) Or Not $aProc[0] Then Return SetError(1, 0, '')
	
	Local $vStruct = DllStructCreate('int[1024]')
	
	Local $hPsapi_Dll = DllOpen('Psapi.dll')
	If $hPsapi_Dll = -1 Then $hPsapi_Dll = DllOpen(@SystemDir & '\Psapi.dll')
	If $hPsapi_Dll = -1 Then $hPsapi_Dll = DllOpen(@WindowsDir & '\Psapi.dll')
	If $hPsapi_Dll = -1 Then Return SetError(2, 0, '')
	
	DllCall($hPsapi_Dll, 'int', 'EnumProcessModules', _
			'hwnd', $aProc[0], _
			'ptr', DllStructGetPtr($vStruct), _
			'int', DllStructGetSize($vStruct), _
			'int*', 0)
	
	Local $aRet = DllCall($hPsapi_Dll, 'int', 'GetModuleFileNameEx', _
			'hwnd', $aProc[0], _
			'int', DllStructGetData($vStruct, 1), _
			'str', '', _
			'int', 2048)
	
	DllClose($hPsapi_Dll)
	
	If Not IsArray($aRet) Or StringLen($aRet[3]) = 0 Then Return SetError(3, 0, '')
	Return $aRet[3]
EndFunc   ;==>__ProcessGetPath

;Check if given path is an existing directory
Func __PathIsFolder($sPath)
	Return StringInStr(FileGetAttrib($sPath), "D") > 0
EndFunc   ;==>__PathIsFolder

;Check if given path is an Opera's Install Dir
Func __PathIsOperaDir($sPath)
	Return FileExists($sPath & "\Opera.exe") = 1
EndFunc   ;==>__PathIsOperaDir

;Check if given path is an Opera's Profile Dir
Func __PathIsOperaProfileDir($sPath)
	Return __PathIsOperaConfigFile($sPath & "\Opera6.ini")
EndFunc   ;==>__PathIsOperaProfileDir

;Check if given path is an Opera's Configuration file
Func __PathIsOperaConfigFile($sPath)
	Local $hFileOpen = FileOpen($sPath, 0)
	If $hFileOpen = -1 Then Return False
	
	Local $sReadLines = FileReadLine($hFileOpen, 1) & FileReadLine($hFileOpen, 2) & FileReadLine($hFileOpen, 3)
	
	FileClose($hFileOpen)
	
	Return FileExists($sPath) And StringRight($sPath, 4) = ".ini" And StringInStr($sReadLines, "Opera Preferences")
EndFunc   ;==>__PathIsOperaConfigFile

;Check if given path is an Opera's Language file
Func __PathIsOperaLangFile($sPath)
	Local $hFileOpen = FileOpen($sPath, 0)
	If $hFileOpen = -1 Then Return False
	
	Local $sReadLines = FileReadLine($hFileOpen, 1) & FileReadLine($hFileOpen, 2) & FileReadLine($hFileOpen, 3)
	
	FileClose($hFileOpen)
	
	Return FileExists($sPath) And StringRight($sPath, 4) = ".lng" And StringInStr($sReadLines, "Opera Language")
EndFunc   ;==>__PathIsOperaLangFile

;Check if given path is an Opera's Skin file
Func __PathIsOperaSkinFile($sPath)
	Return FileExists($sPath) = 1 And StringRight($sPath, 4) = ".zip"
EndFunc   ;==>__PathIsOperaSkinFile

;Check if Prefix string is a supported string to use as Opera's Configuration file
Func __PrefixStrIsOperaConfig($sConfig_Prefix)
	Return StringRegExp($sConfig_Prefix, "(?i)\A(Menu|Toolbar|Keyboard|Mouse)$") = 1
EndFunc   ;==>__PrefixStrIsOperaConfig

;Check the existing of Opera's dialog that prompt about closing Opera Browser, if exists Title of that dialog is returned
Func __ConfirmExitOperaWinExists($sOpDir = -1)
	Local $sLangFile = _OperaGetLangFile($sOpDir)
	If @error Then Return SetError(1, 0, False)
	
	Local $sOpExitTitle = IniRead($sLangFile, "Translation", "1534399195", "")
	Local $sOpActiveTransTitle = IniRead($sLangFile, "Translation", "603725896", "")
	
	If __StringIsUTF8Format($sOpExitTitle) Then $sOpExitTitle = __UTF8ToAnsi($sOpExitTitle)
	If __StringIsUTF8Format($sOpActiveTransTitle) Then $sOpActiveTransTitle = __UTF8ToAnsi($sOpActiveTransTitle)
	
	If $sOpExitTitle <> "" And WinExists($sOpExitTitle) Then Return "[TITLE:" & $sOpExitTitle & "; CLASS:OpWindow]"
	If $sOpActiveTransTitle <> "" And WinExists($sOpActiveTransTitle) Then _
			Return "[TITLE:" & $sOpActiveTransTitle & "; CLASS:OpWindow]"
	
	Return False
EndFunc   ;==>__ConfirmExitOperaWinExists

;Retrieves all runing Opera's PIDs (Process IDs)
Func __GetOperaPIDs()
	Local $aProcList = ProcessList("Opera.exe")
	If @error Then Return SetError(1, 0, 0)
	
	Local $aRetPIDs[1]
	
	For $i = 1 To UBound($aProcList) - 1
		$aRetPIDs[0] += 1
		ReDim $aRetPIDs[$aRetPIDs[0] + 1]
		$aRetPIDs[$aRetPIDs[0]] = $aProcList[$i][1]
	Next
	
	Return $aRetPIDs
EndFunc   ;==>__GetOperaPIDs

;Perform a backup operation for specific file
Func __Backup_Config($sSourceFile, $iReplaceFlag = 0)
	Local $iBackupMode = 0
	
	Switch $iReplaceFlag
		Case 1, 8, 1 + 8
			$iBackupMode = $iReplaceFlag
		Case Else
			$iBackupMode = 0
	EndSwitch
	
	FileCopy($sSourceFile, $sSourceFile & ".bak", $iBackupMode)
EndFunc   ;==>__Backup_Config

;Retrieves new file name for existing file (File (1).txt, File (2).txt etc.)
;by amel27 (A.Melnichuk), mod. by MsCreatoR (G.Sandler)
Func __FileNewName($sFileName, $sDir, $sDelim1 = -1, $sDelim2 = -2)
	If Not __PathIsFolder($sDir) Then Return $sFileName
	If Not FileExists($sDir & "\" & $sFileName) Then Return $sFileName
	
	$sDir = StringRegExpReplace($sDir, "\\ *$", "")
	
	Local $sName = StringRegExpReplace($sFileName, "\.[^.]*$", "")
	Local $sExtn = StringMid($sFileName, StringLen($sName) + 1)
	Local $iCount = 1, $sRet_FileName = $sFileName
	
	While FileExists($sDir & "\" & $sRet_FileName)
		If $sDelim1 = -1 And $sDelim2 = -2 Then
			$sRet_FileName = $sName & " (" & $iCount & ")" & $sExtn
		ElseIf $sDelim1 <> -1 And $sDelim2 <> -2 Then
			$sRet_FileName = $sName & $sDelim1 & $iCount & $sDelim2 & $sExtn
		ElseIf $sDelim1 <> -1 And $sDelim2 = -2 Then
			$sRet_FileName = $sName & $sDelim1 & $iCount & $sExtn
		EndIf
		$iCount += 1
	WEnd
	
	Return $sRet_FileName
EndFunc   ;==>__FileNewName

;Convert UTF-8 String to ANSI string
;by amel27 (A.Melnichuk)
Func __UTF8ToAnsi($sUTF8)
	Local $iLen = StringLen($sUTF8)
	Local $stBuf = DllStructCreate("byte[" & $iLen * 2 & "];byte[2]")
	Local $aRet = DllCall("Kernel32.dll", "int", "MultiByteToWideChar", _
			"int", 65001, "int", 0, _
			"str", $sUTF8, "int", -1, _
			"ptr", DllStructGetPtr($stBuf), "int", $iLen * 2 + 2)
	Local $stOut = DllStructCreate("char[" & $iLen & "];char")
	$aRet = DllCall("kernel32.dll", "int", "WideCharToMultiByte", _
			"int", 0, "int", 0, _
			"ptr", DllStructGetPtr($stBuf), "int", -1, _
			"ptr", DllStructGetPtr($stOut), "int", $iLen + 1, _
			"int", 0, "int", 0)
	Return DllStructGetData($stOut, 1)
EndFunc   ;==>__UTF8ToAnsi

;Check if string contain a UTF-8 format characters
;by amel27 (A.Melnichuk)
Func __StringIsUTF8Format($sString)
	Local $sAsc, $sLen = StringLen($sString), $sExt = $sLen
	For $i = 1 To $sLen
		$sAsc = Asc(StringMid($sString, $i, 1))
		If Not BitAND($sAsc, 0x80) Then
			$sExt = 0
		ElseIf Not BitXOR(BitAND($sAsc, 0xE0), 0xC0) Then
			$sExt = 1
		ElseIf Not (BitXOR(BitAND($sAsc, 0xF0), 0xE0)) Then
			$sExt = 2
		ElseIf Not BitXOR(BitAND($sAsc, 0xF8), 0xF0) Then
			$sExt = 3
		EndIf
		
		If $i + $sExt > $sLen Then Return False
		
		For $j = $i + 1 To $i + $sExt
			$sAsc = Asc(StringMid($sString, $j, 1))
			If BitXOR(BitAND($sAsc, 0xC0), 0x80) Then Return False
		Next
		$i += $sExt
	Next
	Return True
EndFunc   ;==>__StringIsUTF8Format

;Convert string to UTF-8 format
;by LEX1 (A.Ruzanov)
Func __StringToUTF($sString)
	Local $sResult = "", $iCode
	Local $aUTF_Split = StringSplit($sString, "")
	
	For $i = 1 To $aUTF_Split[0]
		$iCode = Asc($aUTF_Split[$i])
		
		Select
			Case $iCode >= 192 And $iCode <= 239
				$aUTF_Split[$i] = Chr(208) & Chr($iCode - 48)
			Case $iCode >= 240 And $iCode <= 255
				$aUTF_Split[$i] = Chr(209) & Chr($iCode - 112)
			Case $iCode = 168
				$aUTF_Split[$i] = Chr(208) & Chr(129)
			Case $iCode = 184
				$aUTF_Split[$i] = Chr(209) & Chr(145)
			Case Else
				$aUTF_Split[$i] = Chr($iCode)
		EndSelect
		
		$sResult &= $aUTF_Split[$i]
	Next
	
	Return $sResult
EndFunc   ;==>__StringToUTF
#EndRegion ================================ HELPER FUNCS ================================
;