#include-once

#Region Header

#CS
	Name: 				AutoItErrorHandler - AutoIt critical error handler.
	Author: 			Copyright  2015 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	AutoIt version: 	3.3.12.0 / 3.3.9.4 / 3.3.8.X / 3.3.6.X
	UDF version:		0.1
	
	Credits:			
						* Firex - SEH.au3 UDF.
	
	Notes:
						* Use supplied GetErrLineCode.au3 to get proper error line code by line number from error that was triggered in compiled script.
						* Memory leaks errors (hard crash and recursion) are not supported.
						* ATM, works only for compiled script.
					
	History:
	
						v0.1
						* First version.
	
#CE

#include <WindowsConstants.au3>
#include <WinAPIProc.au3>
#include <WinAPILocale.au3>
#include <StaticConstants.au3>
#include <ScreenCapture.au3>
#include <GUIConstantsEx.au3>
#include <Date.au3>

#EndRegion Header

#Region Global Variables

Global Enum _
	$iAEH_bSet_ErrLine, $iAEH_bIn_Proc, $iAEH_sMainTitle, $iAEH_sUserFunc, $iAEH_vUserParams, $iAEH_iCOMErrorNumber, $iAEH_sCOMErrorDesc, _
	$iAEH_Total

Global $aAEH_DATA[$iAEH_Total]

#EndRegion Global Variables

#Region User Variables

Global Const $iAEH_CONTINUE_PROC	= 1
Global Const $iAEH_TERMINATE_PROC	= 0

Global Const $iAEH_ButtonsStyle 	= 1
Global Const $nAEH_DefWndBkColor 	= 0xE0DFE2

;===================== Should be changed =====================
Global $sAEH_DevEmailAddress 		= 'myemail@gmail.com' 	;Developer email address
Global $sAEH_DevEmailServer 		= 'smtp.gmail.com' 		;Email server (for gmail it's 'smtp.gmail.com')
Global $sAEH_DevEmailPort 			= 465					;Server port (for gmail it's 465 usualy)
Global $sAEH_DevEmailSSL 			= 1						;SSL enabled (for gmail should be 1)
Global $sAEH_DevEmailUsrName 		= 'myemail'				;Developer email user name
Global $sAEH_DevEmailPass 			= 'mypass'				;Developer email user password
;===================== Should be changed =====================

;Translations
Global $sAEH_Main_Title				= 'AutoIt3 Error'
Global $sAEH_Attention_Title		= 'Attention'
Global $sAEH_Error_Title			= 'Error'
Global $sAEH_Success_Title			= 'Success'

Global $sAEH_ErrMsgSendFrmt_Msg		= 'Program Path: %s\r\n\r\nError Line: %i\r\n\r\nError Description:\r\n\r\n%s\r\n\r\n====================\r\n%s'
Global $sAEH_ErrMsgDispFrmt_Msg		= 'Program has been Terminated :(.\r\nPlease report about this bug to developer, sorry for the inconvenience!\r\n\r\n' & $sAEH_ErrMsgSendFrmt_Msg
Global $sAEH_EnvFrmt_Msg			= 'Environment:\r\n\tAutoIt = %s\r\n\tOSLang = %s\r\n\tKBLayout = %s\r\n\tOS = %s\r\n\tCPU = %s\r\n\tRunTime = %s'
Global $sAEH_MainTxt_Msg			= 'An error occurred in the application.'
Global $sAEH_SendBugReport_Btn		= 'Send bug report'
Global $sAEH_ShowBugReport_Btn		= 'Show bug report'
Global $sAEH_ContinueApp_Btn		= 'Continue application'
Global $sAEH_RestartApp_Btn			= 'Restart application'
Global $sAEH_CloseApp_Btn			= 'Close application'
Global $sAEH_ShowLastScreen_Btn		= 'Show last screen'

;Send bug report part
Global $sAEH_BugReport_Title		= 'Bug Report'
Global $sAEH_SendBugReport_Title	= 'Send Bug Report'
Global $sAEH_ShowLastScreen_Title	= 'Last Screen'
Global $sAEH_SendBugReport_Tip		= 'Please fill the following data (* requierd fields):'
Global $sAEH_EmailServer_Lbl		= '* Email Server:'
Global $sAEH_FromName_Lbl			= '* From name:'
Global $sAEH_FromAddress_Lbl		= '* From address:'
Global $sAEH_ToAddress_Lbl			= '* To address:'
Global $sAEH_Subject_Lbl			= '* Subject:'
Global $sAEH_Body_Lbl				= '* Body (bug report):'
Global $sAEH_SendAttachment_Lbl		= 'Send attachment:'
Global $sAEH_SendingStatus_Lbl		= 'Sending, please wait...'
Global $sAEH_RequierdFields_Msg		= 'Please fill all requierd fields'
Global $sAEH_UnableToSend_Msg		= 'Unable to send bug report, please check the fields data.\n\nError Code:\n\t0x%X\nError Description:\n\t%s'
Global $sAEH_BugReportSent_Msg		= 'Bug Report have been sent.'

#EndRegion User Variables

#Region Public Functions

; #FUNCTION# ====================================================================================================
; Name...........:	_OnAutoItErrorRegister
; Description....:	Registers a function to be called when AutoIt produces a critical error (syntax error usualy).
; Syntax.........:	_OnAutoItErrorRegister( [$sFunction = '' [, $vParams = '' [, $sTitle = '' [, $bUseStdOut = False [, $bSetErrLine = False]]]]])
; Parameters.....:	$sFunction        - [Optional] The name of the user function to call.
;                                           If this parameter is empty (''), then default (built-in) error message function is called.
;                                           Function called with these arguments:
;                                           							$sScriptPath 	- Full path to the script / executable
;                                           							$iScriptLine	- Error script line number
;                                           							$sError_Msg		- Error message
;                                           							$vParams		- User parameters passed by $vParams
;                                           							$hBitmap        - [Optional] If the last window was captured, this will contain hBitmap of captured window (and only if $bUseStdOut = False).
;					$vParams          - [Optional] User defined parameters that passed to $sFunction (default is '' - no parameters).
;					$sTitle           - [Optional] The title of the default error message dialog (used only if $sFunction = '').
;					
; Return values..:	None.
; Author.........:	G.Sandler (CreatoR), www.autoit-script.ru, www.creator-lab.ucoz.ru
; Remarks........:	
; Related........:	_OnAutoItErrorUnRegister
; Example........:	Yes.
; ===============================================================================================================
Func _OnAutoItErrorRegister($sFunction = '', $vParams = '', $sTitle = '')
	If Not @Compiled Then
		Return SetError(-1, 0, 0)
	EndIf
	
	If $aAEH_DATA[$iAEH_bIn_Proc] Then
		If $sFunction <> $aAEH_DATA[$iAEH_sUserFunc] Or $aAEH_DATA[$iAEH_sUserFunc] = '' Then
			_OnAutoItErrorUnRegister()
		Else
			Return ;Prevent conflicts
		EndIf
	EndIf
	
	$aAEH_DATA[$iAEH_bIn_Proc] = True
	$aAEH_DATA[$iAEH_sMainTitle] = ($sTitle ? $sTitle : $sAEH_Main_Title)
	$aAEH_DATA[$iAEH_sUserFunc] = $sFunction
	$aAEH_DATA[$iAEH_vUserParams]	= $vParams
	
	Local $iRet = __AEH_Register(True)
	Return SetError(@error, 0, $iRet)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _OnAutoItErrorUnRegister
; Description ...: UnRegister AutoIt Error Handler.
; Syntax ........: _OnAutoItErrorUnRegister()
; Parameters ....: None.
; Return values .: None.
; Author ........: Firex
; Remarks .......: 
; Related .......: _OnAutoItErrorRegister
; Example .......: Yes.
; ===============================================================================================================================
Func _OnAutoItErrorUnRegister()
	$aAEH_DATA[$iAEH_bIn_Proc] = False
	Return __AEH_Register(False)
EndFunc

#EndRegion Public Functions

#Region Internal Functions

Func __AEH_Register($bRegister)
	Local Static $__Au3ErrCallback[2]
	
    If Not @Compiled Or ((Not $__Au3ErrCallback[0]) <> $bRegister) Then
        Return SetError(-1, 0, False)
	EndIf
	
	Local $aRes, $hModule, $pOffset, $tPatch, $iOpCode_sz, $pvCallback
	Local $Ver, $Arch = 3 + @AutoItX64
	
	Local $aLib[5][5] = [ [4], _ ;* - ptr32
			[ '3.3.12.0', 0x0004CE15, 0x000572F2, _
				'0xFF75ECB8*FFD085C07550909090909090909090', _
				'0x488B4C2430B8*FFD085C0755D909090909090909090909090' _
			], _
			[ '3.3.9.4', 0x00065FAC, 0x000708B0, _
				'0x50B8*FFD085C0754F9090', _
				'0x488B4C2430B8*FFD085C07554909090909090909090909090' _
			], _
			[ '3.3.8.', 0x0005E8ED, 0x0006CE50, _
				2, _ ;Ref to 3.3.9.4
				2 _  ;Ref to 3.3.9.4
			], _
			[ '3.3.6.', 0x0005E7EA, 0x000702B3, _
				'0x50B8*FFD085C0755B9090', _
				'0x488B4C2430B8*FFD085C0756890909090909090909090909090909090' _
			] _
		]
	
	$aRes = DllCall('kernel32.dll', 'handle', 'GetModuleHandle', 'ptr', 0)
	
	If @Error Or Not $aRes[0] Then
		Return SetError(-2, 0, False)
	EndIf
	
	$hModule = $aRes[0]
	
	For $Ver = 1 To $aLib[0][0] Step 1
		If Not StringInStr(@AutoItVersion, $aLib[$Ver][0]) Then
			ContinueLoop
		EndIf
		
		$pOffset = $hModule + Ptr($aLib[$Ver][$Arch-2]) ; !!! Ptr() 3.3.8.1 <= !!!
		
		If IsInt($aLib[$Ver][$Arch]) Then
			$Ver = $aLib[$Ver][$Arch] ;Ref to equal path
		EndIf
		
		$iOpCode_sz = BinaryLen(StringReplace($aLib[$Ver][$Arch], '*', '00000000'))
		
		;Replace opcode region
		$aRes = DllCall('kernel32.dll', 'bool', 'VirtualProtect', 'ptr', $pOffset, 'ulong', $iOpCode_sz, 'dword', 0x40, 'dword*', '')
		
		If @Error Or Not $aRes[0] Then
			ExitLoop
		EndIf

		$tPatch = DllStructCreate('byte Code[' & $iOpCode_sz & ']', $pOffset)
		
		If Not $__Au3ErrCallback[0] Then
			If Not $__Au3ErrCallback[1] Then
				$__Au3ErrCallback[1] = DllStructGetData($tPatch, 'Code') ;Save opcode
			EndIf
			
			$__Au3ErrCallback[0] = DllCallbackRegister('__AEH_OnErrorCallback', 'int', 'ptr')
			$pvCallback = Hex(Binary(DllCallbackGetPtr($__Au3ErrCallback[0])))
			$pvCallback = StringLeft($pvCallback, 8) ;Fix for x64 (dword=4 byte)
			
			DllStructSetData($tPatch, 'Code', StringReplace($aLib[$Ver][$Arch], '*', $pvCallback))
		Else
			DllStructSetData($tPatch, 'Code', $__Au3ErrCallback[1]) ;Restore opcode
			
			DllCallbackFree($__Au3ErrCallback[0])
			$__Au3ErrCallback[0] = 0
		EndIf
		
		DllCall('kernel32.dll', 'bool', 'VirtualProtect', 'ptr', $pOffset, 'ulong', $iOpCode_sz, 'dword', $aRes[4], 'ptr', 0)
		;Replace opcode endregion
		
		Return True
	Next
	
	Return SetError(1, 0, False)
EndFunc

#Region Func _AEH_OnErrorCallback()
Volatile Func __AEH_OnErrorCallback($pErrMsg)
	Local $tError, $aRes
	
	$aRes = DllCall('kernel32.dll', 'int', 'lstrlenW', 'ptr', $pErrMsg)
	
	If @Error Or Not $aRes[0] Then
		Return $iAEH_TERMINATE_PROC ;Terminate script
	EndIf
	
	$tError = DllStructCreate('wchar Msg[' & $aRes[0] & ']', $pErrMsg)
	
	If @Error Then
		Return $iAEH_TERMINATE_PROC ;Terminate script
	EndIf
	
	Local $sError_Msg = DllStructGetData($tError, 'Msg')
	Local $hBitmap = _ScreenCapture_Capture()
	Local $aEnumWin = _WinAPI_EnumWindows()
	
	For $i = 1 To $aEnumWin[0][0]
		If WinGetProcess($aEnumWin[$i][0]) = @AutoItPID And $aEnumWin[$i][1] = 'AutoIt v3 GUI' Then
			_WinAPI_ShowWindow($aEnumWin[$i][0], @SW_HIDE)
		EndIf
	Next
	
	Local $sScriptPath, $iScriptLine
	__AEH_ParseErrorMsg($sScriptPath, $iScriptLine, $sError_Msg)
	
	If @error Then
		Return $iAEH_TERMINATE_PROC ;Terminate script
	EndIf
	
	Local $iRet
	
	If $aAEH_DATA[$iAEH_sUserFunc] = '' Then
		$iRet = __AEH_ShowDefaultErrorDbgMsg($sScriptPath, $iScriptLine, $sError_Msg, $hBitmap)
	Else
		$iRet = Call($aAEH_DATA[$iAEH_sUserFunc], $sScriptPath, $iScriptLine, $sError_Msg, $aAEH_DATA[$iAEH_vUserParams], $hBitmap)
		
		If @error = 0xDEAD And @extended = 0xBEEF Then
			$iRet = Call($aAEH_DATA[$iAEH_sUserFunc], $sScriptPath, $iScriptLine, $sError_Msg, $aAEH_DATA[$iAEH_vUserParams])
		EndIf
	EndIf
	
	Return $iRet ;0 - Terminate script ($iAEH_TERMINATE_PROC), 1 - Continue execute ($iAEH_CONTINUE_PROC)
EndFunc

Func __AEH_ParseErrorMsg(ByRef $sPath, ByRef $iLine, ByRef $sMsg)
	Local $sScriptPath_Pttrn = '(?is)^.*Line \d+\s+\(File "(.*?)"\):\s+.*Error: .*'
	Local $sScriptLine_Pttrn = '(?is)^.*Line (\d+)\s+\(File ".*?"\):\s+.*Error: .*'
	Local $sErrDesc_Pttrn = '(?is)^.*Line \d+\s+\(File ".*?"\):\s+(.*Error: .*)'
	
	If Not StringRegExp($sMsg, $sScriptPath_Pttrn) Then
		Return SetError(1, 0, 0)
	EndIf
	
	$sPath = StringRegExpReplace($sMsg, $sScriptPath_Pttrn, '\1')
	$iLine = StringRegExpReplace($sMsg, $sScriptLine_Pttrn, '\1')
	$sMsg = StringRegExpReplace($sMsg, $sErrDesc_Pttrn, '\1')
	
	$sMsg = StringStripWS(StringRegExpReplace($sMsg, '(?mi)^Error:\h*|:$', ''), 3)
EndFunc

Func __AEH_ShowDefaultErrorDbgMsg($sScriptPath, $iScriptLine, $sErrorMsg, $hBitmap)
	Local $hErrGUI, $nMsg, $SendReport_Button, $ShowBugReport_Button, $ContinueApp_Button, $RestartApp_Button, $CloseApp_Button
	Local $sLastScreen_File = @TempDir & '\AEH_LastScreen.jpg'
	Local $iRet = 0
	
	If $hBitmap Then
		_ScreenCapture_SaveImage($sLastScreen_File, $hBitmap)
	EndIf
	
	$hErrGUI = GUICreate($aAEH_DATA[$iAEH_sMainTitle], 385, 120, -1, -1, BitOR($WS_CAPTION, $WS_POPUP, $WS_SYSMENU), $WS_EX_TOPMOST)
	
	GUISetIcon('User32.dll', -1)
	GUISetBkColor($nAEH_DefWndBkColor)
	
	GUICtrlCreateLabel('', 1, 1, 383, 1)
	GUICtrlSetBkColor(-1, 0x41689E)
	
	GUICtrlCreateLabel('', 1, 118, 383, 1)
	GUICtrlSetBkColor(-1, 0x41689E)
	
	GUICtrlCreateLabel('', 1, 1, 1, 118)
	GUICtrlSetBkColor(-1, 0x41689E)
	
	GUICtrlCreateLabel('', 383, 1, 1, 118)
	GUICtrlSetBkColor(-1, 0x41689E)
	
	GUICtrlCreateIcon('user32.dll', 103, 11, 11, 32, 32)
	
	GUICtrlCreateLabel($sAEH_MainTxt_Msg, 52, 22, 175, 15)
	GUICtrlSetBkColor(-1, -2)
	
	$SendReport_Button = __AEH_CreateButtonEx($sAEH_SendBugReport_Btn, 10, 88, 110, 23, 'shell32.dll', -157, 0xEFEEF2, 0x0000FF) ;, 0x706E63)
	$ShowBugReport_Button = __AEH_CreateButtonEx($sAEH_ShowBugReport_Btn, 125, 88, 115, 23, 'shell32.dll', 23, 0xEFEEF2)
	$ShowLastScreen_Button = __AEH_CreateButtonEx($sAEH_ShowLastScreen_Btn, 245, 88, 130, 23, 'shell32.dll', -118, 0xEFEEF2)
	$ContinueApp_Button = __AEH_CreateButtonEx($sAEH_ContinueApp_Btn, 245, 5, 130, 23, 'shell32.dll', 290, 0xEFEEF2)
	$RestartApp_Button = __AEH_CreateButtonEx($sAEH_RestartApp_Btn, 245, 32, 130, 23, 'shell32.dll', 255, 0xEFEEF2)
	$CloseApp_Button = __AEH_CreateButtonEx($sAEH_CloseApp_Btn, 245, 60, 130, 23, 'shell32.dll', 240, 0xEFEEF2)
	
	If Not $hBitmap Then
		GUICtrlSetState($ShowLastScreen_Button[0], $GUI_DISABLE)
		GUICtrlSetState($ShowLastScreen_Button[1], $GUI_DISABLE)
	EndIf
	
	If FileExists(@WindowsDir & '\Media\chord.wav') Then
		SoundPlay(@WindowsDir & '\Media\chord.wav', 0)
	Else
		DllCall('user32.dll', 'int', 'MessageBeep', 'int', 0x00000010)
	EndIf
	
	GUISetState(@SW_SHOW, $hErrGUI)
	
    While 1
        $nMsg = GUIGetMsg()
		
		If $nMsg = 0 Or ($nMsg > 0 And Not __AEH_ClickProc($nMsg, $hErrGUI)) Then
			ContinueLoop
		EndIf
		
		Switch $nMsg
			Case $SendReport_Button[0], $SendReport_Button[1]
				__AEH_ShowSendEmailGUI($aAEH_DATA[$iAEH_sMainTitle] & ' - ' & $sAEH_SendBugReport_Title, StringFormat($sAEH_ErrMsgSendFrmt_Msg, $sScriptPath, $iScriptLine, $sErrorMsg, __AEH_GetEnvironment()), $hErrGUI, $sLastScreen_File)
			Case $ShowBugReport_Button[0], $ShowBugReport_Button[1]
				If @Compiled Then
					$sScriptPath = @ScriptFullPath
				EndIf
				
				MsgBox(262144 + 4096, $aAEH_DATA[$iAEH_sMainTitle] & ' - ' & $sAEH_BugReport_Title, StringFormat($sAEH_ErrMsgDispFrmt_Msg, $sScriptPath, $iScriptLine, $sErrorMsg, __AEH_GetEnvironment()), 0, $hErrGUI)
			Case $ShowLastScreen_Button[0], $ShowLastScreen_Button[1]
				If $hBitmap And FileExists($sLastScreen_File) Then
					__AEH_ShowLastScreenGUI($sLastScreen_File, $hErrGUI)
				EndIf
			Case $ContinueApp_Button[0], $ContinueApp_Button[1]
				$iRet = 1
				GUIDelete($hErrGUI)
				ExitLoop
			Case $RestartApp_Button[0], $RestartApp_Button[1]
				Local $sRunLine = @AutoItExe & (@Compiled ? '' : ' "' & @ScriptFullPath & '"') & ' ' & $CmdLineRaw
				Run($sRunLine, @ScriptDir)
				ContinueCase
			Case $CloseApp_Button[0], $CloseApp_Button[1], $GUI_EVENT_CLOSE
				GUIDelete($hErrGUI)
				ExitLoop
        EndSwitch
    WEnd
	
	FileDelete($sLastScreen_File)
	Return $iRet
EndFunc

Func __AEH_ShowLastScreenGUI($sScreen_File, $hParent)
	If Not FileExists($sScreen_File) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $hImage, $iWidth, $iHeight, $hScreenGUI, $iPic
	
	_GDIPlus_Startup()
	
	$hImage = _GDIPlus_ImageLoadFromFile($sScreen_File)
	
	If Not $hImage Then
		_GDIPlus_Shutdown()
		Return SetError(1, 0, 0)
	EndIf
	
	$iWidth = _GDIPlus_ImageGetWidth($hImage)
	$iHeight = _GDIPlus_ImageGetHeight($hImage)
	
	_GDIPlus_ImageDispose($hImage)
	_GDIPlus_Shutdown()
	
	If $iWidth >= @DesktopWidth Then
		$iWidth = @DesktopWidth - 100
	EndIf
	
	If $iHeight >= @DesktopHeight - 200 Then
		$iHeight = @DesktopHeight - 200
	EndIf
	
	GUISetState(@SW_DISABLE, $hParent)
	$hScreenGUI = GUICreate($aAEH_DATA[$iAEH_sMainTitle] & ' - ' & $sAEH_ShowLastScreen_Title, $iWidth + 40, $iHeight + 40, -1, -1, -1, $WS_EX_TOOLWINDOW, $hParent)
	
	GUISetIcon('shell32.dll', -118, $hScreenGUI)
	;GUISetBkColor($nAEH_DefWndBkColor, $hScreenGUI)
	
	$iPic = GUICtrlCreatePic($sScreen_File, 20, 20, $iWidth, $iHeight)
	
	GUISetState(@SW_SHOW, $hScreenGUI)
	
	While 1
		Switch GUIGetMsg()
			Case $GUI_EVENT_CLOSE
				GUISetState(@SW_ENABLE, $hParent)
				GUIDelete($hScreenGUI)
				Return
		EndSwitch
	WEnd
EndFunc

Func __AEH_ShowSendEmailGUI($sTitle, $sErrorMsg, $hParent, $sAttachment = '')
	Local $hSndBugRprtGUI, $nServer_Input, $nFromName_Input, $nFromAddress_Input, $nToAddress_Input, $nSubject_Input, $nBody_Edit, $nAttachment_CB, $nAttachment_Lbl
	Local $nSendReport_Label, $nSendReport_Icon, $nStatus_Label, $nMsg, $sServer, $sFromName, $sFromAddress, $sToAddress, $sSubject, $sBody
	Local $iError
	
	GUISetState(@SW_DISABLE, $hParent)
	$hSndBugRprtGUI = GUICreate($sTitle, 400, 350, -1, -1, BitOR($WS_CAPTION, $WS_POPUP, $WS_SYSMENU), -1, $hParent)
	
	GUISetIcon('shell32.dll', -157)
	GUISetBkColor(0xE0DFE2)
	
	GUICtrlCreateLabel('', 1, 1, 398, 1)
	GUICtrlSetBkColor(-1, 0x41689E)
	
	GUICtrlCreateLabel('', 1, 315, 398, 1)
	GUICtrlSetBkColor(-1, 0x41689E)
	
	GUICtrlCreateLabel('', 1, 1, 1, 315)
	GUICtrlSetBkColor(-1, 0x41689E)
	
	GUICtrlCreateLabel('', 398, 1, 1, 315)
	GUICtrlSetBkColor(-1, 0x41689E)
	
	GUICtrlCreateIcon('user32.dll', 103, 11, 11, 32, 32)
	
	GUICtrlCreateLabel($sAEH_MainTxt_Msg & @CRLF & $sAEH_SendBugReport_Tip, 50, 20, 300, 30)
	GUICtrlSetBkColor(-1, -2)
	
	GUICtrlCreateLabel($sAEH_EmailServer_Lbl, 30, 65, -1, 15)
	$nServer_Input = GUICtrlCreateInput($sAEH_DevEmailServer, 30, 80, 150, 20)
	GUICtrlSetState(-1, $GUI_DISABLE)
	
	GUICtrlCreateLabel($sAEH_FromName_Lbl, 230, 65, -1, 15)
	$nFromName_Input = GUICtrlCreateInput(@UserName, 230, 80, 150, 20)
	
	GUICtrlCreateLabel($sAEH_FromAddress_Lbl, 30, 105, -1, 15)
	$nFromAddress_Input = GUICtrlCreateInput('', 30, 120, 150, 20)
	
	GUICtrlCreateLabel($sAEH_ToAddress_Lbl, 230, 105, -1, 15)
	$nToAddress_Input = GUICtrlCreateInput($sAEH_DevEmailAddress, 230, 120, 150, 20)
	GUICtrlSetState(-1, $GUI_DISABLE)
	
	GUICtrlCreateLabel($sAEH_Subject_Lbl, 30, 145, -1, 15)
	$nSubject_Input = GUICtrlCreateInput($sAEH_BugReport_Title & ' - ' & @ScriptName, 30, 160, 350, 20)
	
	GUICtrlCreateLabel($sAEH_Body_Lbl, 30, 185, -1, 15)
	$nBody_Edit = GUICtrlCreateEdit($sErrorMsg, 30, 200, 350, 90)
	
	If $sAttachment <> '' Then
		$nAttachment_CB = GUICtrlCreateCheckbox($sAEH_SendAttachment_Lbl, 30, 295, -1, 15)
		GUICtrlSetState(-1, $GUI_CHECKED)
		$nAttachment_Lbl = GUICtrlCreateLabel(StringRegExpReplace($sAttachment, '^.*\\', ''), 160, 295, -1, 15)
		GUICtrlSetColor(-1, 0x0000FF)
		GUICtrlSetCursor(-1, 0)
		GUICtrlSetFont(-1, 8.5, 200, 4)
		GUICtrlCreateLabel('(' & Round(FileGetSize($sAttachment) / 1024, 1) & ' KB)', 300, 295, 80, 15, $SS_RIGHT)
	EndIf
	
	GUICtrlCreateLabel('', 30, 320, 85, 22)
	GUICtrlSetBkColor(-1, 0x706E63)
	GUICtrlSetState(-1, $GUI_DISABLE)
	
	$nSendReport_Label = GUICtrlCreateLabel($sAEH_SendBugReport_Btn, 34, 324, 80, 15)
	GUICtrlSetBkColor(-1, -2)
	GUICtrlSetColor(-1, 0xFFFFFF)
	GUICtrlSetCursor(-1, 0)
	
	$nSendReport_Icon = GUICtrlCreateIcon('shell32.dll', -157, 13, 323, 16, 16) ;'shimgvw.dll', -6
	GUICtrlSetCursor(-1, 0)
	
	$nStatus_Label = GUICtrlCreateLabel('', 150, 324, 200, 15)
	GUICtrlSetColor(-1, 0xFF0000)
	
	GUISetState(@SW_SHOW, $hSndBugRprtGUI)
	
    While 1
        $nMsg = GUIGetMsg()
		
		If $nMsg = 0 Or ($nMsg > 0 And Not __AEH_ClickProc($nMsg, $hSndBugRprtGUI)) Then
			ContinueLoop
		EndIf
		
		Switch $nMsg
			Case $nAttachment_CB
				If BitAND(GUICtrlRead($nAttachment_CB), $GUI_CHECKED) = $GUI_CHECKED Then
					GUICtrlSetState($nAttachment_Lbl, $GUI_ENABLE)
				Else
					GUICtrlSetState($nAttachment_Lbl, $GUI_DISABLE)
				EndIf
			Case $nAttachment_Lbl
				__AEH_ShowLastScreenGUI($sAttachment, $hSndBugRprtGUI)
			Case $nSendReport_Label, $nSendReport_Icon
				$sServer = GUICtrlRead($nServer_Input)
				$sFromName = GUICtrlRead($nFromName_Input)
				$sFromAddress = GUICtrlRead($nFromAddress_Input)
				$sToAddress = GUICtrlRead($nToAddress_Input)
				$sSubject = GUICtrlRead($nSubject_Input)
				$sBody = GUICtrlRead($nBody_Edit)
				;$sBody = StringSplit(StringStripCR(GUICtrlRead($nBody_Edit)), @LF)
				$sAttach = (BitAND(GUICtrlRead($nAttachment_CB), $GUI_CHECKED) ? $sAttachment : '')
				
				If $sServer = '' Or $sFromName = '' Or $sFromAddress = '' Or $sToAddress = '' Or $sSubject = '' Or $sBody = '' Then
					MsgBox(48, $sAEH_SendBugReport_Title & ' - ' & $sAEH_Attention_Title, $sAEH_RequierdFields_Msg, 0, $hSndBugRprtGUI)
					ContinueLoop
				EndIf
				
				GUICtrlSetData($nStatus_Label, $sAEH_SendingStatus_Lbl)
				GUICtrlSetState($nSendReport_Label, $GUI_DISABLE)
				GUICtrlSetState($nSendReport_Icon, $GUI_DISABLE)
				
				Local $oErrorEvent = ObjEvent('AutoIt.Error', '__AEH_ISMComErrHndlr')
				
				__AEH_INetSmtpMailCom($sServer, $sFromName, $sFromAddress, $sToAddress, $sSubject, $sBody, $sAttach, $sAEH_DevEmailUsrName, $sAEH_DevEmailPass, $sAEH_DevEmailPort, $sAEH_DevEmailSSL)
				
				$iError = @error
				$oErrorEvent = 0
				
				GUICtrlSetData($nStatus_Label, '')
				GUICtrlSetState($nSendReport_Label, $GUI_ENABLE)
				GUICtrlSetState($nSendReport_Icon, $GUI_ENABLE)
				
				If $iError Then
					MsgBox(16, $sAEH_SendBugReport_Title & ' - ' & $sAEH_Error_Title, StringFormat($sAEH_UnableToSend_Msg, $aAEH_DATA[$iAEH_iCOMErrorNumber], $aAEH_DATA[$iAEH_sCOMErrorDesc]), 0, $hSndBugRprtGUI)
				Else
					MsgBox(64, $sAEH_SendBugReport_Title & ' - ' & $sAEH_Success_Title, $sAEH_BugReportSent_Msg, 0, $hSndBugRprtGUI)
				EndIf
				
				$aAEH_DATA[$iAEH_iCOMErrorNumber] = 0
				$aAEH_DATA[$iAEH_sCOMErrorDesc] = ''
			Case $GUI_EVENT_CLOSE
				GUISetState(@SW_ENABLE, $hParent)
				GUIDelete($hSndBugRprtGUI)
				ExitLoop
        EndSwitch
    WEnd
EndFunc

Func __AEH_ClickProc($nCtrlID, $hWnd)
	Local $aCursorInfo
	
	While 1
		$aCursorInfo = GUIGetCursorInfo($hWnd)
		
		If Not @error And $aCursorInfo[2] <> 1 Then
			ExitLoop
		EndIf
	WEnd
	
	If IsArray($aCursorInfo) And $aCursorInfo[4] = $nCtrlID Then
		Return 1
	EndIf
	
	Return 0
EndFunc

Func __AEH_CreateButtonEx($sText, $iLeft, $iTop, $iWidth, $iHeight, $sIconFile = '', $nIconIndex = 0, $nFrameBkColor = -1, $nTxtColor = -1)
	Local $aRet[2]
	
	If $iAEH_ButtonsStyle = 1 Then
		$aRet[0] = GUICtrlCreateIcon($sIconFile, $nIconIndex, $iLeft + 5, $iTop + (($iHeight - 15) / 2), 15, 15)
		$aRet[1] = GUICtrlCreateButton('       ' & $sText & ' ', $iLeft, $iTop, $iWidth, $iHeight, $WS_CLIPSIBLINGS)
		GUICtrlSetBkColor(-1, $nAEH_DefWndBkColor)
	Else
		GUICtrlCreateLabel('', $iLeft, $iTop, $iWidth, $iHeight, $SS_BLACKFRAME)
		GUICtrlSetBkColor(-1, $nFrameBkColor)
		GUICtrlSetState(-1, $GUI_DISABLE)
		
		$aRet[0] = GUICtrlCreateIcon($sIconFile, $nIconIndex, $iLeft + 3, $iTop + (($iHeight - 15) / 2), 15, 15)
		GUICtrlSetCursor(-1, 0)
		
		$aRet[1] = GUICtrlCreateLabel('   ' & $sText, $iLeft + 19, $iTop + 4, $iWidth - 15, $iHeight - 7)
		GUICtrlSetColor(-1, $nTxtColor)
		GUICtrlSetBkColor(-1, $GUI_BKCOLOR_TRANSPARENT)
		GUICtrlSetCursor(-1, 0)
	EndIf
	
	Return $aRet
EndFunc

Func __AEH_GetEnvironment()
	Local $AutoItVer = @AutoItVersion & (@AutoItX64 ? ' x64' : '')
	Local $sOS = @OSVersion & (@OSArch = 'x86' ? '' : ' ' & @OSArch) & ' (' & @OSBuild & (@OSServicePack = '' ? '' : ', Service Pack: ' & @OSServicePack) & ')'
	Return StringFormat($sAEH_EnvFrmt_Msg, $AutoItVer, @OSLang, @KBLayout, $sOS, @CPUArch, __AEH_ProcessGetRunTime(@AutoItPID))
EndFunc

Func __AEH_ProcessGetRunTime($iPID)
	Local $aFT, $tFT, $tST, $iDiff
	
	$aFT = _WinAPI_GetProcessTimes($iPID)
	$tFT = _Date_Time_FileTimeToLocalFileTime(DllStructGetPtr($aFT[0]))
	$tST = _Date_Time_FileTimeToSystemTime(DllStructGetPtr($tFT))
	
	$iDiff = _DateDiff('s', _WinAPI_GetDateFormat(0, $tST, 0, 'yyyy/MM/dd') & ' ' & _WinAPI_GetTimeFormat(0, $tST, $TIME_NOTIMEMARKER), _NowCalc())
	
	Return __AEH_SecondsToTimeStringEx($iDiff, 0409)
EndFunc

Func __AEH_SecondsToTimeStringEx($iSeconds, $iOSLang = @OSLang, $iFlag = 2, $sDelim = ', ')
	Local $sYear, $sYears, $s5Years
	Local $sDay, $sDays, $s5Days
	Local $sHour, $sHours, $s5Hours
	Local $sMin, $sMins, $s5Mins
	Local $sSec, $sSecs, $s5Secs
	
	If StringInStr('0419', $iOSLang) Then ;Russian support
		Dim $sYear = '', $sYears = '', $s5Years = ''
		Dim $sDay = '', $sDays = '', $s5Days = ''
		Dim $sHour = '', $sHours = '', $s5Hours = ''
		Dim $sMin = '', $sMins = '', $s5Mins = ''
		Dim $sSec = '', $sSecs = '', $s5Secs = ''
	Else
		Dim $sYear = 'Year', $sYears = 'Years', $s5Years = 'Years'
		Dim $sDay = 'Day', $sDays = 'Days', $s5Days = 'Days'
		Dim $sHour = 'Hour', $sHours = 'Hours', $s5Hours = 'Hours'
		Dim $sMin = 'Minute', $sMins = 'Minutes', $s5Mins = 'Minutes'
		Dim $sSec = 'Second', $sSecs = 'Seconds', $s5Secs = 'Seconds'
	EndIf
	
	If Number($iSeconds) >= 0 Then
		Local $iTicks = Int($iSeconds / 3600), $iDays, $iYears
		Local $iHours = Mod($iTicks, 24)
		
		If $iTicks >= 24 Then
			$iDays = ($iTicks - $iHours) / 24
		EndIf
		
		If $iDays >= 365 Or BitAND($iFlag, 2) = 0 Then
			$iYears = Int($iDays / 365)
			
			If ($iYears >= 20 And StringRight($iYears, 1) >= 2) Or ($iYears < 20 And $iYears >= 2) Then
				$sYear = $sYears
			EndIf
			
			If ($iYears >= 20 And (StringRight($iYears, 1) >= 5 Or StringRight($iYears, 1) < 1)) Or ($iYears < 20 And $iYears >= 5) Or $iYears = 0 Then
				$sYear = $s5Years
			EndIf
			
			$iYears &= ' ' & $sYear & $sDelim
		EndIf
		
		If $iDays > 0 Or BitAND($iFlag, 2) = 0 Then
			$iDays = Mod($iDays, 365)
			
			If ($iDays >= 20 And StringRight($iDays, 1) >= 2) Or ($iDays < 20 And $iDays >= 2) Then
				$sDay = $sDays
			EndIf
			
			If ($iDays >= 20 And (StringRight($iDays, 1) >= 5) Or StringRight($iDays, 1) < 1) Or ($iDays < 20 And $iDays >= 5) Or $iDays = 0 Then
				$sDay = $s5Days
			EndIf
			
			$iDays &= ' ' & $sDay & $sDelim
		EndIf
		
		$iTicks = Mod($iSeconds, 3600)
		
		Local $iMins = Int($iTicks / 60)
		Local $iSecs = Round(Mod($iTicks, 60))
		
		If BitAND($iFlag, 1) = 1 Then
			If StringLen($iHours) = 1 Then
				$iHours = '0' & $iHours
			EndIf
			
			If StringLen($iMins) = 1 Then
				$iMins = '0' & $iMins
			EndIf
			
			If StringLen($iSecs) = 1 Then
				$iSecs = '0' & $iSecs
			EndIf
		EndIf
		
		If ($iHours >= 20 And StringRight($iHours, 1) >= 2) Or ($iHours < 20 And $iHours >= 2) Then
			$sHour = $sHours
		EndIf
		If ($iHours >= 20 And (StringRight($iHours, 1) >= 5 Or StringRight($iHours, 1) < 1)) Or ($iHours < 20 And $iHours >= 5) Or $iHours = 0 Then
			$sHour = $s5Hours
		EndIf
		If ($iMins >= 20 And StringRight($iMins, 1) >= 2) Or ($iMins < 20 And $iMins >= 2) Then
			$sMin = $sMins
		EndIf
		If ($iMins >= 20 And (StringRight($iMins, 1) >= 5 Or StringRight($iMins, 1) < 1)) Or ($iMins < 20 And $iMins >= 5) Or $iMins = 0 Then
			$sMin = $s5Mins
		EndIf
		If ($iSecs >= 20 And StringRight($iSecs, 1) >= 2) Or ($iSecs < 20 And $iSecs >= 2) Then
			$sSec = $sSecs
		EndIf
		If ($iSecs >= 20 And (StringRight($iSecs, 1) >= 5 Or StringRight($iSecs, 1) < 1)) Or ($iSecs < 20 And $iSecs >= 5) Or $iSecs = 0 Then
			$sSec = $s5Secs
		EndIf
		
		If BitAND($iFlag, 2) = 2 Then
			If Number($iHours) = 0 Then
				$iHours = ''
			Else
				$iHours &= ' ' & $sHour & $sDelim
			EndIf
			
			If Number($iMins) = 0 Then
				$iMins = ''
			Else
				$iMins &= ' ' & $sMin & $sDelim
			EndIf
		Else
			$iHours &= ' ' & $sHour & $sDelim
			$iMins &= ' ' & $sMin & $sDelim
		EndIf
		
		$iSecs &= ' ' & $sSec
		
		Return $iYears & $iDays & $iHours & $iMins & $iSecs
	EndIf
	
	Return SetError(1, 0, 0)
EndFunc

Func __AEH_INetSmtpMailCom($s_SmtpServer, $s_FromName, $s_FromAddress, $s_ToAddress, $s_Subject = '', $s_Body = '', $s_Attachment = '', $s_Username = '', $s_Password = '', $IPPort = 25, $ssl = 0)
	Local $objEmail = ObjCreate('CDO.Message')
	If @error Or Not IsObj($objEmail) Then Return SetError(1, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
	
	If FileExists($s_Attachment) And $s_Body = '' Then
		Return SetError(2, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
	EndIf
	
    $objEmail.From = '"' & $s_FromName & '" <' & $s_FromAddress & '>'
	If @error Then Return SetError(3, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
	
    $objEmail.To = $s_ToAddress
	If @error Then Return SetError(4, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
	
	$objEmail.Subject = $s_Subject
	If @error Then Return SetError(5, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
   
	If StringInStr($s_Body, '<') And StringInStr($s_Body, '>') Then
        $objEmail.HTMLBody = $s_Body
    Else
        $objEmail.Textbody = $s_Body & @CRLF
    EndIf
    
	If @error Then Return SetError(6, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
	
	If FileExists($s_Attachment) Then
		$objEmail.AddAttachment($s_Attachment)
		If @error Then Return SetError(7, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
	EndIf
	
	$objEmail.Configuration.Fields.Item('http://schemas.microsoft.com/cdo/configuration/sendusing') = 2
	If @error Then Return SetError(8, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
	
    $objEmail.Configuration.Fields.Item('http://schemas.microsoft.com/cdo/configuration/smtpserver') = $s_SmtpServer
	If @error Then Return SetError(9, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
	
    $objEmail.Configuration.Fields.Item('http://schemas.microsoft.com/cdo/configuration/smtpserverport') = $IPPort
   If @error Then Return SetError(10, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
   
   If $s_Username <> '' Then
        $objEmail.Configuration.Fields.Item('http://schemas.microsoft.com/cdo/configuration/smtpauthenticate') = 1
		If @error Then Return SetError(11, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
		
        $objEmail.Configuration.Fields.Item('http://schemas.microsoft.com/cdo/configuration/sendusername') = $s_Username
		If @error Then Return SetError(12, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
		
        $objEmail.Configuration.Fields.Item('http://schemas.microsoft.com/cdo/configuration/sendpassword') = $s_Password
		If @error Then Return SetError(13, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
    EndIf
   
	If $ssl Then
        $objEmail.Configuration.Fields.Item('http://schemas.microsoft.com/cdo/configuration/smtpusessl') = True
		If @error Then Return SetError(14, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
    EndIf
   
	$objEmail.Configuration.Fields.Update
	If @error Then Return SetError(15, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
	
    $objEmail.Send
	If @error Then Return SetError(16, 0, $aAEH_DATA[$iAEH_sCOMErrorDesc])
EndFunc

Func __AEH_ISMComErrHndlr($oError)
	$aAEH_DATA[$iAEH_iCOMErrorNumber] = $oError.Number
	$aAEH_DATA[$iAEH_sCOMErrorDesc] = StringStripWS(StringStripWS($oError.WinDescription, 3) & ' ' & StringStripWS($oError.Description, 3), 3)
	
	Return SetError(1, 0, 0)
EndFunc

#EndRegion Internal Functions
